import Widget from '../Widget'
import axios from 'axios'
import LIVR from 'livr'

LIVR.Validator.defaultAutoTrim();

export default class Form extends Widget {
    start(options) {
        if (!options.form)
            throw (new Error('not found form'))
        this.form = document.querySelector(options.form)
        this.url = options.url

        this.optionalFields = []
        this.defaultFields = []

        this.defaultRules = {}
        this.optionalRules = {}

        this.confirmButton = this.form.querySelector('button[type="submit"]')
        this.captcha = false;
        this.errors = {};

        if (options.captcha) {
            this.captcha = true;
            this.form.querySelector('#renewCaptcha').addEventListener('click', (e) => this.renewCaptcha(e))
        }

        if (options.errors) {
            this.errors = options.errors
        }

        if (options.rating) {
            $(options.rating).barrating({
                theme: 'css-stars',
                showSelectedRating: false
            })
        }

        this.successUrl = options.successUrl ? options.successUrl : '/mlogin'

        if (this.confirmButton)
            this.confirmButton.addEventListener('click', (e) => this.submitForm(e))

        if (options.optional) {
            [...this.form.querySelectorAll(`input[name=${options.optionalControl}]`)]
                .map(input => input.addEventListener('change', ({target}) => this.toggleOptional(target.value === '1')))
        }

        this.getFields()
    }

    toggleOptional(show) {
        this.resetErrors()
        this.optionalRules = {}

        this.optionalFields.map(field => {
            field.classList.toggle('optional--active')
            if (field.dataset.validation && show)
                this.addRule(field.getAttribute('name'), field.dataset.validation, true)
        })
    }

    submitForm(e) {
        e.preventDefault();

        this.resetErrors();

        const values = this.getValues();

        const validator = this.getValidator(Object.assign({}, this.defaultRules, this.optionalRules))

        const valid = validator.validate(values);

        if (valid) {
            const submitData = window.Object.assign({}, values, valid)
            this.sendData(submitData)
        } else {
            this.handleValidationErrors(validator.getErrors());
        }
    }

    sendData(data) {
        axios({
            url: this.url,
            method: 'post',
            data: data,
            transformRequest: [(data) => {
                const form = new FormData;
                for (const formItem in data) {
                    form.append(formItem, data[formItem])
                }
                return form
            }]
        })
            .then(({data}) => {
                if (data.valid) {
                    if (data.redirect) {
                        return location.href = data.redirect;
                    }
                    return location.href = this.successUrl
                } else {
                    this.handleValidationErrors(data.errors)
                }
            })
            .catch(err => console.log(err))
    }

    getFields() {
        this.fields = this.form.querySelectorAll('input[type=text], input[type=password], input[type=checkbox], input[type=radio], input[type=hidden], textarea, select')

        const fields = this.form.querySelectorAll('input[type=text], input[type=password], input[type=checkbox], input[type=radio], input[type=hidden], textarea, select')

        ;[...fields].map(field => {
            if (!field.dataset.optional) {
                this.defaultFields.push(field)
                if (field.dataset.validation) {
                    this.addRule(field.getAttribute('name'), field.dataset.validation)
                }
            } else {
                this.optionalFields.push(field)
            }
        })
    }

    addRule(name, rules, optional = false) {
        if (typeof rules !== 'object')
            rules = JSON.parse(rules)

        if (optional) return this.optionalRules = window.Object.assign({}, this.optionalRules, {[name]: rules})

        this.defaultRules = window.Object.assign({}, this.defaultRules, {[name]: rules})
    }

    getValidator(rules) {
        return new LIVR.Validator(rules)
    }

    getValues() {
        const values = {};

        [...this.fields].map(field => {
            if (field.getAttribute('type') === 'checkbox') {
                return values[field.getAttribute('name')] = field.checked ? 1 : ''
            }

            if (field.getAttribute('type') === 'radio') {
                if (!field.checked) {
                    return
                }

                return values[field.getAttribute('name')] = field.value
            }

            values[field.getAttribute('name')] = field.value
        })

        return values;
    }

    handleValidationErrors(errors) {
        const fields = {}

        const errorsI18n = {
                'WRONG_EMAIL': 'Неправильный email адрес',
                'TOO_SHORT': 'Слишком короткий пароль',
                'REQUIRED': 'Необходимо заполнить поле',
                'required' : 'Необходимо заполнить поле',
                'not.correct': 'Неправильный логин или пароль',
                'not.unique': 'Такой email уже зарегистрирован',
                'not.phone': 'Такой телефонный номер уже зарегистрирован',
                'sms.code': 'Неверный код подтверждения',
                'match': 'Пароль и подтверждение не совпадают',
                'not.valid': 'Аккаунт находится на модерации'
            }

        ;[...this.fields].map(field => {
            fields[field.getAttribute('name')] = field
        })

        for (const error in errors) {
            const errorElement = window.document.createElement('span')
            errorElement.classList.add('error')
            errorElement.style.color = 'red'

            if (this.errors[errors[error]]) {
                errorElement.innerHTML = this.errors[errors[error]]
            } else if (errorsI18n[errors[error]]) {
                errorElement.innerHTML = errorsI18n[errors[error]]
            } else {
                errorElement.innerHTML = errors[error]
            }

            fields[error].parentNode.insertBefore(errorElement, fields[error])
            fields[error].style.border = '2px solid red'
        }

        const y = this.form.querySelector('.error').scrollTop;
        window.scrollTo(0, y);
        if (this.captcha) this.renewCaptcha();
    }

    resetErrors() {
        [...this.form.querySelectorAll('.error')].map(error => error.remove())
        ;[...this.form.querySelectorAll('input')].map(input => input.style.border = '2px solid #9CC6EE')
    }

    renewCaptcha(e = null) {
        if (e)
            e.preventDefault()

        this.form.querySelector('.captcha input[type=text]').value = ''
        this.form.querySelector('.captcha img').src = `/captcha?rnd=${Math.random()}`
    }
}