package envoy

import (
	"sync"
	"time"
)

const (
	defaultWindowLength time.Duration = 10 * time.Second
	updateBufferLen                   = 8
)

type StateChange struct {
	Condition Condition
	Message   string
	changedAt time.Time
}

type StateUpdates map[*State]*StateChange

type Envoy struct {
	sync.RWMutex
	service      string
	states       []*State
	windowLength time.Duration
	transitions  chan struct{}
	updates      chan StateUpdates
}

var (
	startTime time.Time = time.Now()
)

// HTTPServerConf configures envoy to start an HTTP server.
type HTTPServerConf struct {
	Host string
	Port string
}

// NagiosConf configures envoy to send passive notifications to Nagios.
// It calls the send_nsca binary on the host.
// This configuration must be given the Host and Port of the nagios server.
// It also needs the name of the Service check as defined in Nagios
// and the FQDN of your host.
type NagiosConf struct {
	Host        string
	Port        string
	Service     string
	Node        string
	NSCATimeout time.Duration
	NSCAPath    string
}

// Config is the top-level configuration for envoy.
// You must give it the name of your service, which will be reported whenever
// envoy is queried.
type Config struct {
	Service      string          // The name of the service.
	WindowLength time.Duration   // (optional) The window length used for flap mitigation. If unspecified will default to 10s.
	NscaConf     *NagiosConf     // (optional) Config for nagios passive checks. If unspecified no passive check will be sent.
	ServerConf   *HTTPServerConf // (optional) Config for service state server. If unspecified no server will start.
}

func uptime() int {
	return int(time.Since(startTime).Seconds())
}

func (e *Envoy) applyUpdates(updates StateUpdates) {
	e.Lock()
	defer e.Unlock()

	for state, change := range updates {
		state.update(change)
	}
}

func (e *Envoy) listen() {
	for updates := range e.updates {
		e.applyUpdates(updates)
	}
}

func (e *Envoy) Update(updates StateUpdates) {
	now := time.Now()
	for _, update := range updates {
		update.changedAt = now
	}

	e.updates <- updates
}

func (e *Envoy) confInit(config *Config) {
	e.service = config.Service

	if config.WindowLength > time.Duration(0) {
		e.windowLength = config.WindowLength
	}

	if config.NscaConf != nil {
		go e.runPusher(config.NscaConf)
	}

	if config.ServerConf != nil {
		go e.runServer(config.ServerConf)
	}
}

func NewEnvoy(config *Config) *Envoy {
	e := &Envoy{
		states:       make([]*State, 0),
		transitions:  make(chan struct{}),
		updates:      make(chan StateUpdates, updateBufferLen),
		windowLength: defaultWindowLength,
	}

	go e.listen()

	if config != nil {
		e.confInit(config)
	}

	return e
}
