package envoy

import (
	"encoding/json"
	"fmt"
	"sort"
	"strings"
	"time"

	"code.justin.tv/common/golibs/pkgpath"
)

const globalStateDesc string = "Global state"

var (
	buildSha string = ""
	mainPkg  string = ""
	build    *buildInfo
)

type history map[Condition]time.Time

func (h history) MarshalJSON() ([]byte, error) {
	history := make(map[string]int)
	for c, t := range h {
		history[c.String()] = int(time.Since(t).Seconds())
	}

	return json.Marshal(history)
}

// A StateSnapshot holds a fixed, immutable snapshot of the values from a state together
// with the string and json rendering definitions.
type StateSnapshot struct {
	Name        string    `json:"-"`
	Description string    `json:"description"`
	Message     string    `json:"message"`
	Current     Condition `json:"current"`
	History     history   `json:"history"`
}

func (s StateSnapshot) String() string {
	return fmt.Sprintf("%s: %s; %s", s.Current, s.Description, s.Message)
}

type buildInfo struct {
	Sha string `json:"sha,omitempty"`
	Pkg string `json:"pkg,omitempty"`
}

func (b *buildInfo) String() string {
	s := []string{}
	if b.Pkg != "" {
		s = append(s, fmt.Sprintf("Pkg: %s", b.Pkg))
	}
	if b.Sha != "" {
		s = append(s, fmt.Sprintf("SHA: %s", b.Sha))
	}
	return strings.Join(s, " ")
}

func globalSnapshot(snapshots map[string]StateSnapshot) StateSnapshot {
	worst := ConditionNormal
	worstCount := 0
	for _, s := range snapshots {
		if s.Current == worst {
			worstCount++
		} else if s.Current.IsWorse(worst) {
			worst = s.Current
			worstCount = 1
		}
	}
	pluralizer := "states"
	if worstCount == 1 {
		pluralizer = "state"
	}

	return StateSnapshot{
		Description: globalStateDesc,
		Message:     fmt.Sprintf("%d %s", worstCount, pluralizer),
		Current:     worst,
		History:     history{},
	}
}

// stateSnapshots is responsible for managing a collection of stateSnapshot's and be able
// to render the data as both a human readable string representation and a computer consumable
// JSON representation
type stateSnapshots struct {
	Uptime         time.Duration
	Snapshots      map[string]StateSnapshot
	GlobalSnapshot StateSnapshot
	Service        string
}

func (s stateSnapshots) String() string {
	data := make([]string, 0)
	data = append(data, fmt.Sprintf("INFO: Service: %s", s.Service))
	if build != nil {
		data = append(data, fmt.Sprintf("INFO: Build: %s", build.String()))
	}
	data = append(data, fmt.Sprintf("INFO: Uptime: %ds", int(s.Uptime.Seconds())))

	data = append(data, s.GlobalSnapshot.String())

	sorted := make(sort.StringSlice, 0, len(s.Snapshots))
	for name := range s.Snapshots {
		sorted = append(sorted, name)
	}
	sort.Sort(sorted)

	for _, name := range sorted {
		data = append(data, s.Snapshots[name].String())
	}

	data = append(data, "\n")

	return strings.Join(data, "\n")
}

func (s stateSnapshots) MarshalJSON() ([]byte, error) {
	data := struct {
		Service string                   `json:"service"`
		Uptime  int                      `json:"uptime"`
		Current string                   `json:"current"`
		States  map[string]StateSnapshot `json:"states"`
		Build   *buildInfo               `json:"build,omitempty"`
	}{
		s.Service,
		int(s.Uptime.Seconds()),
		s.GlobalSnapshot.Current.String(),
		s.Snapshots,
		build,
	}

	return json.Marshal(data)
}

func (e *Envoy) newStateSnapshots() stateSnapshots {
	e.RLock()
	defer e.RUnlock()
	snapshots := make(map[string]StateSnapshot)
	for _, s := range e.states {
		snapshots[s.name] = s.Snapshot()
	}
	return stateSnapshots{
		Uptime:         time.Since(startTime),
		Snapshots:      snapshots,
		GlobalSnapshot: globalSnapshot(snapshots),
		Service:        e.service,
	}
}

func init() {
	maybeMainPkg, ok := pkgpath.Main()
	if ok {
		mainPkg = maybeMainPkg
	}
	if buildSha != "" || mainPkg != "" {
		build = &buildInfo{
			Sha: buildSha,
			Pkg: mainPkg,
		}
	}
}
