package logger

// Logger provides a simple interface for logging messages in a sane way by
// providing a standard message format as well as log levels.
//
// To use simply call logger.Error, logger.Warn, logger.Info, or logger.Debug
// with the arguments $calling_function, $message_fmt, and the subsequent
// arguments
// Example: logger.Info("main", "starting server: %d", time.Now().Unix())
// Example: logger.Error("main", "got error getting config: %s", err.Error())
// Example: logger.Debug("main", "here is a message")
//
// Note: to kill a program with log.Fatal call logger.Fatal($function, error)
//
// To set the LogLevel of the program use logger.SetLogLevel($level)
// where you can use one of the provided levels (LogError, LogWarn, LogInfo,
// LogDebug) or you can call logger.SetLogLevelString($string) with a level
// in the form of debug, info, warn, or error
//

import (
	"fmt"
	"log"
	"net/http"
	"strings"
)

const (
	LogError LogLevel = iota
	LogWarn
	LogInfo
	LogDebug

	// log level strings
	logLevelError = "error"
	logLevelWarn  = "warning"
	logLevelInfo  = "info"
	logLevelDebug = "debug"

	logsLimit = 400
)

var (
	logLevel    LogLevel = LogInfo
	logMessages *logs
)

func Init() {
	logMessages = newLogs()
	http.HandleFunc("/logger", loggerHandler)
}

type LogLevel int

func Fatal(function string, err error) {
	write(function, "fatal", "got fatal err: %s", err.Error())
	log.Fatal(err)
}

func Error(function string, message string, v ...interface{}) {
	if logLevel >= LogError {
		write(function, logLevelError, message, v...)
	}
}

func Warn(function string, message string, v ...interface{}) {
	if logLevel >= LogWarn {
		write(function, logLevelWarn, message, v...)
	}
}

func Info(function string, message string, v ...interface{}) {
	if logLevel >= LogInfo {
		write(function, logLevelInfo, message, v...)
	}
}

func Debug(function string, message string, v ...interface{}) {
	if logLevel == LogDebug {
		write(function, logLevelDebug, message, v...)
	}
}

func write(function string, level string, format string, v ...interface{}) {
	logMsg := fmt.Sprintf("[%s] [%s] %s", function, level, fmt.Sprintf(format, v...))
	switch {
	case fileLog == true:
		fileLogBuff <- logMsg
	default:
		log.Print(logMsg)
	}
	if logMessages != nil {
		logMessages.add(logMsg)
	}
}

func GetLogLevel() string {
	switch {
	case logLevel == LogError:
		return logLevelError
	case logLevel == LogWarn:
		return logLevelWarn
	case logLevel == LogInfo:
		return logLevelInfo
	case logLevel == LogDebug:
		return logLevelDebug
	default:
		return "unknown"
	}
}

func SetLogLevel(level LogLevel) {
	Info("SetLogLevel", "changing log level to: %v", level)
	// don't set if level doesn't make sense
	if level > 3 {
		return
	}
	logLevel = level
}

func SetLogLevelString(level string) {
	level = strings.ToLower(level)
	switch {
	case level == "error":
		logLevel = LogError
	case level == "warn":
		logLevel = LogWarn
	case level == "debug":
		logLevel = LogDebug
	default:
		logLevel = LogInfo
	}
}

type logs struct {
	ct   int
	buff chan string
	msgs chan string
}

func (l *logs) run() {
	for {
		s, ok := <-l.buff
		if !ok {
			panic("logger.logs.run: channel closed")
		}
		l.procLog(s)
	}
}

func (l *logs) add(s string) {
	l.buff <- s
}

func (l *logs) procLog(s string) {
	select {
	case l.msgs <- s:
	default:
		<-l.msgs
		l.msgs <- s
	}
}

func (l *logs) get() []string {
	ss := make([]string, 0)
	for i := 0; i < logsLimit; i++ {
		select {
		case s := <-l.msgs:
			ss = append(ss, s)
		default:
			break
		}
	}
	return ss
}

func newLogs() *logs {
	l := &logs{
		ct:   0,
		buff: make(chan string, logsLimit),
		msgs: make(chan string, logsLimit),
	}
	go l.run()
	return l
}

func loggerHandler(w http.ResponseWriter, r *http.Request) {
	ss := logMessages.get()
	s := ""
	for i := range ss {
		s = s + fmt.Sprintf("%s\n", ss[i])
	}
	_, err := w.Write([]byte(s))
	if err != nil {
		Error("loggerHandler", "got err writing response: %v", err)
	}
	return
}

func LogLevelFromString(level string) LogLevel {
	switch {
	case level == logLevelError:
		return LogError
	case level == logLevelWarn:
		return LogWarn
	case level == logLevelInfo:
		return LogInfo
	case level == logLevelDebug:
		return LogDebug
	default:
		return LogInfo
	}
}
