
-- Line chart showing total current activations by anchor location (fullscreen overlay, component overlay, panel), daily

WITH
the_constants_timeframe AS (
    SELECT
      '2017-05-01'::TIMESTAMP AS range_start
      , DATE_TRUNC('day', (SYSDATE - INTERVAL '7 hours')) + INTERVAL '1 day' AS range_end
      --, '2018-06-18'::TIMESTAMP AS start_time --this is when we had component extension activation test data
      , '2018-06-11'::TIMESTAMP AS start_time -- ryan requested more historical data https://jira.twitch.com/browse/DSDS-268
      , DATE_TRUNC('day', (SYSDATE - INTERVAL '7 hours')) AS the_date_today
)
, the_constants_released_extensions AS (
  SELECT extension_id, extension_name, extension_category_id, is_bits_enabled, is_component_enabled, is_overlay_enabled, is_panel_enabled, is_whitelisted_extension
  FROM analysis.extensions_metadata_full AS emf
  WHERE the_date = (SELECT the_date_today FROM the_constants_timeframe)
  AND extension_state='Released'
  GROUP BY 1,2,3,4,5,6,7,8
)

--used as table to left join the activation/deactivation/uninstall/delete logs against
, the_basetable_date AS (
  SELECT *
  FROM (
    SELECT
      (SELECT range_start FROM the_constants_timeframe) AS start_date
      , date::timestamp AS end_date
    FROM metadata.date
    WHERE date>=(SELECT range_start FROM the_constants_timeframe) AND date<=(SELECT range_end FROM the_constants_timeframe)
  )
  WHERE end_date>(SELECT start_time FROM the_constants_timeframe)
)


--get raw extension_activate logs
, activate_logs_step0 AS (
  SELECT base."time", 'extension_activate' AS the_event
    , base.user_id, base.extension_id, base.extension_version, base.extension_anchor
  FROM tahoe.extension_activate AS base
  INNER JOIN the_constants_released_extensions AS rel ON base.extension_id = rel.extension_id
  WHERE base.date>=(SELECT range_start FROM the_constants_timeframe)
  AND (
    (base.extension_anchor = 'component' AND rel.is_component_enabled AND base.time>='2018-06-16'::TIMESTAMP) --if extension_anchor = component then needs to be component enabled extension
    OR (base.extension_anchor = 'video_overlay' AND rel.is_overlay_enabled AND base.time>='2018-06-16'::TIMESTAMP) --if extension_anchor = overlay then needs to be overlay enabled extension
    OR (base.extension_anchor = 'panel' AND rel.is_panel_enabled AND base.time>='2018-06-16'::TIMESTAMP) --if extension_anchor = panel then needs to be panel enabled extension
    OR (base.time<'2018-06-16'::TIMESTAMP) --before this date we didn't have valid extension_anchor event logging for extension_activate
  )
  AND base.extension_version IS NOT NULL
)
--join the raw extension_activate logs against our base table that caps the logs to up to end of each day
, activate_logs_step1 AS (
  SELECT start_date, end_date, extension_id, user_id, extension_version, extension_anchor, MAX(time) AS last_time_activate
  FROM (
    SELECT base.start_date, base.end_date, ea.time, ea.extension_id, ea.extension_version, ea.user_id, ea.extension_anchor
    FROM the_basetable_date AS base
    LEFT JOIN activate_logs_step0 AS ea ON base.start_date<=ea.time AND ea.time<base.end_date
  )
  GROUP BY 1,2,3,4,5,6
)
--keep only the records of their latest activate for each user_id and extension_id combo
, activate_logs_step2 AS (
  SELECT *
  FROM (
    SELECT *, MAX(last_time_activate) OVER (PARTITION BY start_date, end_date, user_id, extension_id) AS max_last_time_activate
    FROM activate_logs_step1
  )
  WHERE last_time_activate = max_last_time_activate
)

--get raw extension_deactivate logs
, deactivate_logs_step0 AS (
  SELECT "time", 'extension_deactivate' AS the_event, user_id, extension_id, extension_version
  FROM tahoe.extension_deactivate
  WHERE date>=(SELECT range_start FROM the_constants_timeframe)
  AND extension_id IN (SELECT extension_id FROM the_constants_released_extensions)
  AND extension_version IS NOT NULL
)
--join the raw extension_deactivate logs against our base table that caps the logs to up to end of each day
, deactivate_logs_step1 AS (
  SELECT start_date, end_date, extension_id, user_id, extension_version, MAX(time) AS last_time_deactivate
  FROM (
    SELECT base.start_date, base.end_date, ea.time, ea.extension_id, ea.extension_version, ea.user_id
    FROM the_basetable_date AS base
    LEFT JOIN deactivate_logs_step0 AS ea ON base.start_date<=ea.time AND ea.time<base.end_date
  )
  GROUP BY 1,2,3,4,5
)
--keep only the records of their latest deactivate for each user_id and extension_id combo
, deactivate_logs_step2 AS (
  SELECT *
  FROM (
    SELECT *, MAX(last_time_deactivate) OVER (PARTITION BY start_date, end_date, user_id, extension_id) AS max_last_time_deactivate
    FROM deactivate_logs_step1
  )
  WHERE last_time_deactivate = max_last_time_deactivate
)

--uninstall by channel is an implicit deactivate
--get raw extension_uninstall logs
, uninstall_logs_step0 AS (
  SELECT "time", 'extension_uninstall' AS the_event, user_id, extension_id, extension_version
  FROM tahoe.extension_uninstall
  WHERE date>=(SELECT range_start FROM the_constants_timeframe)
  AND extension_id IN (SELECT extension_id FROM the_constants_released_extensions)
  AND extension_version IS NOT NULL
)
--join the raw extension_uninstall logs against our base table that caps the logs to up to end of each day
, uninstall_logs_step1 AS (
  SELECT start_date, end_date, extension_id, user_id, extension_version, MAX(time) AS last_time_uninstall
  FROM (
    SELECT base.start_date, base.end_date, ea.time, ea.extension_id, ea.extension_version, ea.user_id
    FROM the_basetable_date AS base
    LEFT JOIN uninstall_logs_step0 AS ea ON base.start_date<=ea.time AND ea.time<base.end_date
  )
  GROUP BY 1,2,3,4,5
)
--keep only the records of their latest deactivate for each user_id and extension_id combo
, uninstall_logs_step2 AS (
  SELECT *
  FROM (
    SELECT *, MAX(last_time_uninstall) OVER (PARTITION BY start_date, end_date, user_id, extension_id) AS max_last_time_uninstall
    FROM uninstall_logs_step1
  )
  WHERE last_time_uninstall = max_last_time_uninstall
)

--deletion by developer is an implicit deactivate across all channels
--get raw extension_uninstall logs
, delete_logs_step0 AS (
  SELECT "time", 'extension_delete' AS the_event, extension_id, extension_version
  FROM tahoe.extension_delete
  WHERE date>=(SELECT range_start FROM the_constants_timeframe)
  AND extension_id IN (SELECT extension_id FROM the_constants_released_extensions)
  AND extension_version IS NOT NULL
)
--join the raw extension_delete logs against our base table that caps the logs to up to end of each day
, delete_logs_step1 AS (
  SELECT start_date, end_date, extension_id, extension_version, MAX(time) AS last_time_delete
  FROM (
    SELECT base.start_date, base.end_date, ea.time, ea.extension_id, ea.extension_version
    FROM the_basetable_date AS base
    LEFT JOIN delete_logs_step0 AS ea ON base.start_date<=ea.time AND ea.time<base.end_date
  )
  GROUP BY 1,2,3,4
)
--keep only the records of their latest delete for each extension_id combo
, delete_logs_step2 AS (
  SELECT * FROM delete_logs_step1
)

--join the logs to get activate funnel
, the_activate_funnel_step1 AS (
  SELECT base.end_date - INTERVAL '1 day' AS the_day
    , base.extension_id, base.user_id, base.extension_anchor
    , base.last_time_activate, deact.last_time_deactivate
    , unin.last_time_uninstall, del.last_time_delete
  FROM activate_logs_step2 AS base
  LEFT JOIN deactivate_logs_step2 AS deact
    ON base.start_date=deact.start_date AND base.end_date=deact.end_date
    AND base.extension_id = deact.extension_id AND base.user_id = deact.user_id
  LEFT JOIN uninstall_logs_step2 AS unin
    ON base.start_date=unin.start_date AND base.end_date=unin.end_date
    AND base.extension_id = unin.extension_id AND base.user_id = unin.user_id
  LEFT JOIN delete_logs_step2 AS del
    ON base.start_date=del.start_date AND base.end_date=del.end_date
    AND base.extension_id = del.extension_id AND base.extension_version=del.extension_version
)
--add logic to determine if the extension is activated on that day
, the_activate_funnel_step2 AS (
  SELECT *
    , CASE
      WHEN last_time_activate IS NULL THEN false  --if no activate timestamp then can't be activated
      WHEN last_time_deactivate >= last_time_activate AND last_time_deactivate IS NOT NULL THEN false
      WHEN last_time_uninstall >= last_time_activate AND last_time_uninstall IS NOT NULL THEN false
      WHEN last_time_delete >= last_time_activate AND last_time_delete IS NOT NULL THEN false
      ELSE true
    END AS bool_extension_activated
  FROM the_activate_funnel_step1
)
--join the data from activate and install funnel together
, the_logs_final AS (
  SELECT *, NVL(extension_anchor_new_tracking, extension_anchor_old) AS extension_anchor
  FROM (
    SELECT base.the_day, base.extension_anchor AS extension_anchor_new_tracking, rel.extension_anchor AS extension_anchor_old, base.user_id, base.extension_id
    FROM the_activate_funnel_step2 AS base
    LEFT JOIN analysis.released_extensions_latest AS rel ON base.extension_id = rel.extension_id
    WHERE base.bool_extension_activated
  )
)

SELECT the_day, extension_anchor
--, net_activated_extensions::FLOAT / distinct_users::FLOAT AS extensions_per_user
--, distinct_users
, net_activated_extensions
FROM (
  SELECT the_day, extension_anchor
    , COUNT(DISTINCT base.user_id) AS distinct_users
    , COUNT(DISTINCT base.user_id+base.extension_id) AS net_activated_extensions
  FROM the_logs_final AS base
  GROUP BY 1,2
)
ORDER BY 1 DESC, 2


-- --select the final joined data
-- SELECT the_day, extension_anchor, extension_anchor_old, NVL(extension_anchor, extension_anchor_old) AS final_extension_anchor
-- --, net_activated_extensions::FLOAT / distinct_users::FLOAT AS extensions_per_user
-- --, distinct_users
-- , net_activated_extensions
-- FROM (
--   SELECT base.the_day, base.extension_anchor, rel.extension_anchor AS extension_anchor_old
--     , COUNT(DISTINCT base.user_id) AS distinct_users
--     , COUNT(DISTINCT base.user_id+base.extension_id) AS net_activated_extensions
--   FROM the_logs_final AS base
--   LEFT JOIN analysis.released_extensions_latest AS rel ON base.extension_id = rel.extension_id
--   WHERE base.bool_extension_activated
--   GROUP BY 1,2,3
-- )
-- ORDER BY 1 DESC, 2
