package combiner

import (
	"testing"
	"time"

	"code.justin.tv/release/trace/events"
	"github.com/golang/protobuf/proto"
)

const (
	megabyte  = 1000000
	flushTime = 100 * time.Millisecond
)

var typicalRecord = newPbrecord(0, 160) // average record size is 160bytes

// Protobuf Record & RecordSet
type pbrecord events.Event

func newPbrecord(key uint64, size int) Record {
	if size < 20 {
		return &pbrecord{
			TransactionId: []uint64{key, 0},
		}
	}
	return &pbrecord{
		TransactionId: []uint64{key, 0},
		// empirically, about 20 bytes of data elsewhere in the record
		Hostname: string(make([]byte, size-20)),
	}
}

func (r *pbrecord) Key() int {
	return int(r.TransactionId[0])
}

func (r *pbrecord) NewSet() RecordSet {
	return newPbrecordset()
}

type pbrecordset struct {
	events *events.EventSet
	buf    *proto.Buffer
}

func newPbrecordset() *pbrecordset {
	return &pbrecordset{
		events: &events.EventSet{},
		buf:    proto.NewBuffer(nil),
	}
}

func (rs *pbrecordset) Push(r Record) {
	rs.events.Event = append(rs.events.Event, (*events.Event)(r.(*pbrecord)))
	rs.buf.Marshal(&events.EventSet{Event: []*events.Event{(*events.Event)(r.(*pbrecord))}})
}

func (rs *pbrecordset) Pop() Record {
	var r *events.Event
	r, rs.events.Event = rs.events.Event[len(rs.events.Event)-1], rs.events.Event[:len(rs.events.Event)-1]
	rs.buf.Reset()
	rs.buf.Marshal(rs.events)
	return (*pbrecord)(r)
}

func (rs *pbrecordset) Encode() []byte {
	return rs.buf.Bytes()
}

func (rs *pbrecordset) Size() int {
	return len(rs.buf.Bytes())
}

func BenchmarkRecordCombinerAddProto(b *testing.B) {
	rc := NewRecordCombiner(1, megabyte, flushTime)

	// drain output
	go func() {
		for _ = range rc.Blobs() {
		}
	}()

	go func() {
		for err := range rc.Errors() {
			b.Fatalf("err=%q", err)
		}
	}()

	b.ResetTimer()
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			rc.Add(typicalRecord)
		}
	})
	b.StopTimer()
	rc.Close()
}

func BenchmarkShardAddProto(b *testing.B) {
	ch := make(chan *Blob, b.N)
	errs := make(chan error, b.N)
	s := newShard("", megabyte, flushTime, ch, errs)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		s.add(typicalRecord)
	}
	b.StopTimer()
	s.close()
}

func BenchmarkShardProcessProto(b *testing.B) {
	ch := make(chan *Blob, b.N)
	errs := make(chan error, b.N)
	s := newShard("", megabyte, flushTime, ch, errs)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		s.process(typicalRecord)
	}
	b.StopTimer()
	s.close()
}
