package main

import (
	"fmt"
	"math"
	"strings"
)

type FlavorScore struct {
	MilliScore          int
	Prefix              string
	Suffix              string
	lastEmoteMilliScore int
}

// toMessage returns the canonical text representation of a given score
func (f *FlavorScore) toMessage() string {
	message := ""
	if f.Prefix != "" {
		message += f.Prefix + " "
	}

	message += numberString(f)

	if f.Suffix != "" {
		if f.Suffix == "repeating" {
			message += "...x"
		} else {
			message += "x"
		}
		message += " " + f.Suffix
	} else {
		message += "x"
	}

	message += " :flavortown:"

	return message
}

// Make a very specific text version of a number
// Probably could be replaced by a fancy sprintf
func numberString(f *FlavorScore) string {
	numberString := ""
	negative := f.MilliScore < 0
	absScore := f.MilliScore
	if negative {
		absScore *= -1
		numberString += "-"
	}
	mainScore := absScore / 1000
	deciScore := absScore % 1000
	numberString += fmt.Sprintf("%d", mainScore)
	f1 := deciScore / 100
	if f1 != 0 {
		numberString += fmt.Sprintf(".%d", f1)
		f2 := deciScore % 100 / 10
		if f2 != 0 {
			numberString += fmt.Sprintf("%d", f2)
			f3 := deciScore % 10
			if f3 != 0 {
				numberString += fmt.Sprintf("%d", f3)
			}
		}
	}

	return numberString
}

// Caclculates the score for an emote and adds it to the flavorescore
// TODO: Handle 50/50 emotes generically?
// TODO: handle dance emotes generically?
func (f *FlavorScore) addEmote(emote string, first bool) {
	// Strip colons
	// so switches are just on strings even if there are prefixes
	if strings.HasPrefix(emote, ":") {
		emote = emote[1:]
	}
	if strings.HasSuffix(emote, ":") {
		emote = emote[:len(emote)-1]
	}

	switch emote {
	// Plain emotes
	case "flavortown":
		f.MilliScore += 1000
		f.lastEmoteMilliScore = 1000
	case "noflavortown":
		f.MilliScore -= 1000
		f.lastEmoteMilliScore = -1000
	case "fieriparrot":
		f.MilliScore += 7500
		f.lastEmoteMilliScore = 7500
	case "spicy":
		f.MilliScore += 4000
		f.Prefix = "solid"
		f.lastEmoteMilliScore = 4000
	case "partyparrot":
		f.MilliScore += 4000
		f.lastEmoteMilliScore = 4000
	case "notbad":
		f.MilliScore += 3888
		f.Suffix = "repeating"
		f.lastEmoteMilliScore = 3888
	case "ok_hand":
		f.MilliScore += 3750
		f.lastEmoteMilliScore = 3750
	case "+1":
		f.MilliScore += 3456
		f.lastEmoteMilliScore = 3456
	case "seemsgood":
		f.MilliScore += 3247
		f.Prefix = "roughly"
		f.lastEmoteMilliScore = 3247
	case "harambae":
		f.MilliScore += 1122
		f.Suffix = "repeating"
		f.lastEmoteMilliScore = 1122
	case "ceelo":
		if first {
			f.MilliScore += 0
			f.lastEmoteMilliScore = 0
		} else {
			score := f.lastEmoteMilliScore + 500
			f.MilliScore += score
			f.lastEmoteMilliScore = score
		}
	case "bezos":
		f.MilliScore += 500
		f.lastEmoteMilliScore = 500
	case "negativeflavortown":
		// Basically negaflavortown but full negative allowed
		f.MilliScore -= 1000
		f.lastEmoteMilliScore = -1000

	// 50/50 emotes
	case "flavorbae":
		f.MilliScore += 1061
		f.Suffix = "repeating"
		f.lastEmoteMilliScore = 1061

	// Dance emotes
	case "hypermegaflavordance":
		// Note: Dance emotes can't be done generically in this system
		// Just calculate this one explicitly
		score := int(math.Floor(math.Pow(3, math.Phi) * 1000))
		f.MilliScore += score
		f.lastEmoteMilliScore = score

	default:
		if strings.HasPrefix(emote, "nega") {
			f.addEmote(emote[4:], first)
			f.MilliScore -= 2 * f.lastEmoteMilliScore
			f.lastEmoteMilliScore *= -1
		} else if strings.HasPrefix(emote, "maybe") {
			f.addEmote(emote[5:], first)
			f.MilliScore -= f.lastEmoteMilliScore / 2
			f.lastEmoteMilliScore /= 2
		} else if strings.HasPrefix(emote, "hyper") {
			f.addEmote(emote[5:], first)
			f.MilliScore += f.lastEmoteMilliScore
			f.lastEmoteMilliScore *= 2
		} else if strings.HasPrefix(emote, "mega") {
			f.addEmote(emote[4:], first)
			f.MilliScore += f.lastEmoteMilliScore / 2
			f.lastEmoteMilliScore = f.lastEmoteMilliScore * 3 / 2
		} else if strings.HasPrefix(emote, "omega") {
			f.addEmote(emote[5:], first)
			f.MilliScore += 4 * f.lastEmoteMilliScore
			f.lastEmoteMilliScore *= 5
		} else if strings.HasPrefix(emote, "onega") {
			f.addEmote(emote[5:], first)
			f.MilliScore -= 6 * f.lastEmoteMilliScore
			f.lastEmoteMilliScore *= -5
		} else if strings.HasPrefix(emote, "golden") {
			f.addEmote(emote[6:], first)
			f.MilliScore += 4 * f.lastEmoteMilliScore
			f.lastEmoteMilliScore *= 5
		} else {
			// TODO dance, combo
			fmt.Println("Unhandled emote found: ", emote)
		}
	}

}
