defmodule Cluster.Strategy.DNS do
  @moduledoc """
  This clustering strategy works by using :inet_tcp.getaddrs to look up all of the ip
  addresses associated with each host name. For each address, it will call
  Node.connect using the given node name, defaulting to "erlang" for the node
  name. It will continually monitor connections every :polling_interval
  millisconds. It assumes that all nodes are using longnames.

  An example configuration is below:


      config :libcluster,
	topologies: [
	  dns_example: [
	    strategy: #{__MODULE__},
	    config: [
	      connect_nodes: "elixir@app1 elixir@app2 elixir@172.16.5.30",
	      polling_interval: 10_000]]]

  """
  use GenServer
  use Cluster.Strategy
  import Cluster.Logger

  alias Cluster.Strategy.State

  @default_polling_interval 5_000
  @default_node_name "erlang"

  def start_link(opts), do: GenServer.start_link(__MODULE__, opts)
  def init(opts) do
    state = %State{
      topology: Keyword.fetch!(opts, :topology),
      connect: Keyword.fetch!(opts, :connect),
      disconnect: Keyword.fetch!(opts, :disconnect),
      config: Keyword.fetch!(opts, :config),
      meta: MapSet.new([])
    }
    {:ok, state, 0}
  end

  def handle_info(:timeout, state) do
    handle_info(:load, state)
  end
  def handle_info(:load, %State{topology: topology, connect: connect, disconnect: disconnect} = state) do
    new_nodelist = MapSet.new(get_nodes(state))
    added        = MapSet.difference(new_nodelist, state.meta)
    removed      = MapSet.difference(state.meta, new_nodelist)
    new_nodelist = case Cluster.Strategy.disconnect_nodes(topology, disconnect, MapSet.to_list(removed)) do
		:ok ->
		  new_nodelist
		{:error, bad_nodes} ->
		  # Add back the nodes which should have been removed, but which couldn't be for some reason
		  Enum.reduce(bad_nodes, new_nodelist, fn {n, _}, acc ->
		    MapSet.put(acc, n)
		  end)
	      end
    new_nodelist = case Cluster.Strategy.connect_nodes(topology, connect, MapSet.to_list(added)) do
	      :ok ->
		new_nodelist
	      {:error, bad_nodes} ->
		# Remove the nodes which should have been added, but couldn't be for some reason
		Enum.reduce(bad_nodes, new_nodelist, fn {n, _}, acc ->
		  MapSet.delete(acc, n)
		end)
	    end
    Process.send_after(self(), :load, Keyword.get(state.config, :polling_interval, @default_polling_interval))
    {:noreply, %{state | :meta => new_nodelist}}
  end
  def handle_info(_, state) do
    {:noreply, state}
  end

  @spec get_nodes(State.t) :: [atom()]
  defp get_nodes(%State{topology: topology, config: config}) do
    nodes =
      config
      |> Keyword.get(:connect_nodes)
      |> String.split
      |> Enum.map(fn node -> expand_nodes(String.split(node, "@")) ; end)

    for [{:error, reason}] <- nodes, do: warn(topology, reason)
    for [{:ok, node_spec}] <- nodes, do: String.to_atom(node_spec)
  end

  @spec expand_nodes([String.t]) :: [String.t]
  defp expand_nodes([node_host]), do: expand_nodes(["erlang", node_host])
  defp expand_nodes([node_name, node_host]) do
    case :inet_tcp.getaddrs(String.to_charlist(node_host)) do
      {:ok, ips} ->
	for {a,b,c,d} <- ips do
	  {:ok, "#{node_name}@#{a}.#{b}.#{c}.#{d}"}
	end

      {:error, reason} ->
	  [{:error, "Error resolving #{inspect node_host}: #{inspect reason}"}]
    end
  end
end
