package main

import (
	"errors"
	"flag"
	"log"
	"net"
	"strconv"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/jsonconf"
	"code.justin.tv/marketing/altlookup/app/server"
	"code.justin.tv/marketing/altlookup/clients"
	"code.justin.tv/marketing/altlookup/config"
)

func main() {
	confPath := flag.String("conf", "", "Path to config file")
	flag.Parse()

	conf, err := parseConfig(*confPath)
	if err != nil {
		log.Fatalln("error parsing config:", err)
	}

	clients, err := setupClients(conf)
	if err != nil {
		log.Fatalln("error setting up clients:", err)
	}
	s := server.NewServer(conf, clients)

	params := gojiplus.ServerParams{
		Port:      conf.Port,
		DebugPort: conf.DebugPort,
		Stats:     clients.Stats,
	}
	if err := gojiplus.ListenAndServe(s.Handler, params); err != nil {
		log.Fatalln("Error starting server:", err)
	}
}

func setupClients(conf *config.Config) (*server.Clients, error) {
	statsHostPort := hostport(conf.Stats.Host, conf.Stats.Port)
	stats, err := statsd.NewBufferedClient(statsHostPort, conf.Stats.Prefix, 1*time.Second, 512)
	if err != nil {
		return nil, errx.New(err)
	}

	sitedb, err := clients.NewSiteDB(conf.SiteDB, "sitedb", stats)
	if err != nil {
		return nil, errx.New(err)
	}

	return &server.Clients{
		Stats:  stats,
		SiteDB: sitedb,
	}, nil
}

func parseConfig(path string) (*config.Config, error) {
	if path == "" {
		return nil, errors.New("Empty config path")
	}

	conf := &config.Config{}
	err := jsonconf.ReadFile(conf, path)
	if err != nil {
		return nil, err
	}
	return conf, nil
}

func hostport(host string, port int) string {
	if port == 0 {
		return host
	}
	return net.JoinHostPort(host, strconv.Itoa(port))
}
