package server

import (
	"encoding/json"
	"net/http"
	"strconv"

	"github.com/cactus/go-statsd-client/statsd"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/logx"
	"code.justin.tv/marketing/altlookup/clients"
	"code.justin.tv/marketing/altlookup/config"
)

type Clients struct {
	Stats  statsd.Statter
	SiteDB clients.SiteDB
}

type Server struct {
	http.Handler
	config  *config.Config
	clients *Clients
}

func NewServer(config *config.Config, clients *Clients) *Server {
	s := &Server{
		config:  config,
		clients: clients,
	}

	root := goji.NewMux()
	root.Handle(pat.Get("/debug/running"), gojiplus.HealthCheck())

	mux := goji.SubMux()
	root.HandleC(pat.New("/*"), mux)
	mux.HandleFuncC(pat.Get("/v1/alts"), s.GetUserAlts)

	s.Handler = root
	return s
}

// serveJSON writes a JSON response.
func serveJSON(w http.ResponseWriter, r *http.Request, v interface{}) {
	content, err := json.Marshal(v)
	if err != nil {
		http.Error(w, gojiplus.UnknownErrorString, http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json")
	if _, err = w.Write(content); err != nil {
		http.Error(w, gojiplus.UnknownErrorString, http.StatusInternalServerError)
		return
	}
}

// TODO: We should be able to deduce the right status based on the error type.
func serveError(ctx context.Context, w http.ResponseWriter, r *http.Request, err error, status int) {
	w.WriteHeader(status)

	// Log to rollbar and logs.
	fields := logx.Fields{
		"method": r.Method,
		"status": status,
		"ip":     gojiplus.GetIPFromRequest(r),
		"url":    r.URL,
		"err":    err,
	}
	if gojiplus.IsServerError(status) {
		logx.Error(ctx, err, fields)
	} else {
		logx.Warn(ctx, err, fields)
	}
}
