package clients

import (
	"database/sql"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/dbplus"
	"code.justin.tv/chat/errx"
	"code.justin.tv/chat/tmi/clue/util"
	"code.justin.tv/marketing/altlookup/config"
	"code.justin.tv/marketing/altlookup/definitions"
)

const statsInterval = 5 * time.Second

type SiteDB interface {
	AlternateAccountsOfUser(ctx context.Context, userID string) ([]definitions.AccountInfo, error)
	AlternateAccountsOfEmail(ctx context.Context, userID, email string) ([]definitions.AccountInfo, error)
	AlternateAccountsOfIP(ctx context.Context, userID, ip string) ([]definitions.AccountInfo, error)
}

func NewSiteDB(conf config.DBConfig, xactGroup string, stats statsd.Statter) (SiteDB, error) {
	logger := dbplus.NewLogger(stats, conf.StatsPrefix, 1.0, conf.LogQueries)
	db, err := dbplus.OpenDB(dbplus.Config{
		Host: conf.Host,
		Port: conf.Port,

		Name:     conf.Name,
		User:     conf.User,
		Password: conf.Password,

		MaxOpenConns:       conf.MaxOpenConns,
		MaxIdleConns:       conf.MaxIdleConns,
		ConnQueueSize:      conf.ConnQueueSize,
		ConnAcquireTimeout: time.Duration(conf.ConnAcquireTimeout),
		RequestTimeout:     time.Duration(conf.RequestTimeout),
		MaxConnAge:         time.Duration(conf.MaxConnAge),

		LogQueries:  conf.LogQueries,
		StatsPrefix: conf.StatsPrefix,
	})
	if err != nil {
		return nil, err
	}
	db.SetCallbacks(logger.LogDBStat, logger.LogRunStat)

	querier := dbplus.NewQuerier("sitedb", xactGroup, logger)

	manager := dbplus.NewManager(db, logger)
	go manager.Manage(statsInterval)

	return &siteDBImpl{
		db:      db,
		querier: querier,
		manager: manager,
	}, nil
}

type siteDBImpl struct {
	db      db.DB
	querier *dbplus.DbQuerier
	manager *dbplus.DbManager
}

func (sdb *siteDBImpl) AlternateAccountsOfUser(ctx context.Context, userID string) ([]definitions.AccountInfo, error) {
	var altAccounts []definitions.AccountInfo
	queryID := "alternate_accounts_of_user"
	query := `SELECT u.id, u.login, LOWER(u.email), u.remote_ip, u.created_on, terms_of_service_violation FROM users u WHERE u.id = $1 LIMIT 500`
	rows, err := sdb.db.Query(ctx, queryID, query, userID)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	for rows.Next() {
		var id int
		var login string
		var email sql.NullString
		var remoteIP sql.NullString
		var createdOn util.NullTime
		var termsOfServiceViolation sql.NullBool
		if err := rows.Scan(&id, &login, &email, &remoteIP, &createdOn, &termsOfServiceViolation); err != nil {
			return nil, err
		}
		if !email.Valid {
			email.String = ""
		}
		if !remoteIP.Valid {
			remoteIP.String = ""
		}
		if !termsOfServiceViolation.Valid {
			termsOfServiceViolation.Bool = false
		}
		if !createdOn.Valid {
			createdOn.Time = time.Now().UTC()
		}
		altAccounts = append(altAccounts, definitions.AccountInfo{
			UserID: strconv.Itoa(id),
			Login:  login,
			Email:  email.String,
			IP:     remoteIP.String,
			TermsOfServiceViolation: termsOfServiceViolation.Bool,
			CreatedOn:               createdOn.Time,
			ParentAccountID:         userID,
			ParentAccountMatchType:  "initial",
		})
	}
	return altAccounts, nil
}

func (sdb *siteDBImpl) AlternateAccountsOfEmail(ctx context.Context, userID, email string) ([]definitions.AccountInfo, error) {
	if email == "" {
		return nil, nil
	}
	email = strings.ToLower(email)
	var altAccounts []definitions.AccountInfo
	queryID := "alternate_accounts_of_email"
	query := `SELECT u.id, u.login, LOWER(u.email), u.remote_ip, u.created_on, terms_of_service_violation FROM users u WHERE LOWER(u.email) = $1 LIMIT 500`
	rows, err := sdb.db.Query(ctx, queryID, query, email)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	for rows.Next() {
		var id int
		var login string
		var email sql.NullString
		var remoteIP sql.NullString
		var createdOn util.NullTime
		var termsOfServiceViolation sql.NullBool
		if err := rows.Scan(&id, &login, &email, &remoteIP, &createdOn, &termsOfServiceViolation); err != nil {
			return nil, err
		}
		if !email.Valid {
			email.String = ""
		}
		if !remoteIP.Valid {
			remoteIP.String = ""
		}
		if !termsOfServiceViolation.Valid {
			termsOfServiceViolation.Bool = false
		}
		if !createdOn.Valid {
			createdOn.Time = time.Now().UTC()
		}
		altAccounts = append(altAccounts, definitions.AccountInfo{
			UserID: strconv.Itoa(id),
			Login:  login,
			Email:  email.String,
			IP:     remoteIP.String,
			TermsOfServiceViolation: termsOfServiceViolation.Bool,
			CreatedOn:               createdOn.Time,
			ParentAccountID:         userID,
			ParentAccountMatchType:  "email",
		})
	}
	return altAccounts, nil
}

func (sdb *siteDBImpl) AlternateAccountsOfIP(ctx context.Context, userID, ip string) ([]definitions.AccountInfo, error) {
	if ip == "" {
		return nil, nil
	}
	var altAccounts []definitions.AccountInfo
	queryID := "alternate_accounts_of_ip"
	query := `SELECT u.id, u.login, LOWER(u.email), u.remote_ip, u.created_on, terms_of_service_violation FROM users u WHERE u.remote_ip = $1 LIMIT 500`
	rows, err := sdb.db.Query(ctx, queryID, query, ip)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	for rows.Next() {
		var id int
		var login string
		var email sql.NullString
		var remoteIP sql.NullString
		var createdOn util.NullTime
		var termsOfServiceViolation sql.NullBool
		if err := rows.Scan(&id, &login, &email, &remoteIP, &createdOn, &termsOfServiceViolation); err != nil {
			return nil, err
		}
		if !email.Valid {
			email.String = ""
		}
		if !remoteIP.Valid {
			remoteIP.String = ""
		}
		if !termsOfServiceViolation.Valid {
			termsOfServiceViolation.Bool = false
		}
		if !createdOn.Valid {
			createdOn.Time = time.Now().UTC()
		}
		altAccounts = append(altAccounts, definitions.AccountInfo{
			UserID: strconv.Itoa(id),
			Login:  login,
			Email:  email.String,
			IP:     remoteIP.String,
			TermsOfServiceViolation: termsOfServiceViolation.Bool,
			CreatedOn:               createdOn.Time,
			ParentAccountID:         userID,
			ParentAccountMatchType:  "ip",
		})
	}
	return altAccounts, nil
}

func (sdb *siteDBImpl) wrapErr(err error, queryID string) error {
	if err == nil {
		return nil
	}
	return errx.New(err, errx.Fields{
		"db":       "sitedb",
		"query_id": queryID,
	})
}
