package dbplus

import (
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/logx"
)

type DbLogger struct {
	stats           statsd.Statter
	statsPrefix     string
	statsSampleRate float32
	logQueries      bool
}

func NewLogger(stats statsd.Statter, statsPrefix string, statsSampleRate float32, logQueries bool) *DbLogger {
	return &DbLogger{
		stats:           stats,
		statsPrefix:     statsPrefix,
		statsSampleRate: statsSampleRate,
		logQueries:      logQueries,
	}
}

func (l *DbLogger) LogDBStat(evName, queryName string, duration time.Duration) {
	l.stats.TimingDuration(l.stat("query."+queryName+"."+evName), duration, l.statsSampleRate)
}

func (l *DbLogger) LogRunStat(evName string) {
	l.stats.Inc(l.stat("dbevent."+evName), 1, l.statsSampleRate)
}

func (l *DbLogger) printQuery(ctx context.Context, dbName, query string, params []interface{}) {
	if l.logQueries {
		logx.Info(ctx, "tmi db query", logx.Fields{
			"db_name": dbName,
			"query":   query,
			"params":  params,
		})
	}
}

func (l *DbLogger) stat(stat string) string {
	return l.statsPrefix + stat
}

func (l *DbLogger) logDBState(info db.DBInfo) {
	l.stats.Gauge(l.stat("state.open_conns_cap"), int64(info.OpenConnsCap), 1.0)
	l.stats.Gauge(l.stat("state.max_open_conns"), int64(info.MaxOpenConns), 1.0)
	l.stats.Gauge(l.stat("state.min_available_conns"), int64(info.MinAvailableConns), 1.0)
}
