package errx

import (
	"fmt"

	"github.com/stvp/rollbar"
)

const (
	stackBufSize = 1024
)

var _ error = (*errImpl)(nil)
var _ Error = (*errImpl)(nil)

type Error interface {
	Error() string
	String() string
	Fields() Fields
}

type Fields map[string]interface{}

type errImpl struct {
	err    error
	fields Fields
}

// New accepts an arbitrary type, appends arbitrary metadata, and returning a *errx.Error type.
// If val is:				then return:
// 		nil						nil
// 		a *errx.errImpl			the same error to avoid another allocation
// 		a native error			a wrapped error
// 		anything else			a wrapped error with the Go string representation
func New(val interface{}, fields ...Fields) Error {
	return NewWithSkip(val, 1, fields...)
}

func NewWithSkip(val interface{}, skip int, fields ...Fields) Error {
	var err *errImpl
	switch e := val.(type) {
	case nil:
		return nil
	case *errImpl:
		err = e
	case error:
		err = &errImpl{
			err: e,
		}
	default:
		err = &errImpl{
			err: fmt.Errorf("%v", e),
		}
	}

	// Include rollbar-compatible stack trace by default
	if err.fields == nil {
		err.fields = map[string]interface{}{
			"stack": rollbar.BuildStack(2 + skip),
		}
	}

	for _, f := range fields {
		for k, v := range f {
			err.fields[k] = v
		}
	}
	return err
}

func (e *errImpl) Error() string  { return e.err.Error() }
func (e *errImpl) String() string { return e.err.Error() }
func (e *errImpl) Fields() Fields { return e.fields }

// Error unwraps an error (either native or *errx.errImpl) and returns a native Go error.
func Unwrap(err error) error {
	if e, ok := err.(*errImpl); ok {
		return e.err
	}
	return err
}
