package util

import (
	"errors"
	"strconv"
	"strings"
)

type ErrSlice []error

func (errs ErrSlice) String() string {
	return errs.Error()
}

func (errs ErrSlice) Error() string {
	strs := make([]string, len(errs))
	for i, err := range errs {
		strs[i] = err.Error()
	}
	return strings.Join(strs, "; ")
}

type BoolError struct {
	Bool  bool
	Error error
}

// Runs input functions in parallel
// Signals result as soon as one of the functions returns a true result without error
func OrAsync(fns ...func() (bool, error)) chan BoolError {
	fnDone := make(chan BoolError, len(fns))

	done := make(chan BoolError)

	for _, fn := range fns {
		go func(fn func() (bool, error)) {
			b, err := fn()
			fnDone <- BoolError{b, err}
		}(fn)
	}

	go func() {
		r := BoolError{false, nil}
		for i := 0; i < len(fns); i++ {
			result := <-fnDone
			if result.Error != nil {
				r = result
				continue
			}
			if result.Bool {
				done <- result
				return
			}
		}
		done <- r
	}()

	return done
}

func StringInSlice(needle string, haystack []string) bool {
	for _, s := range haystack {
		if s == needle {
			return true
		}
	}
	return false
}

// ParseIntSlice parses a string (delimited by `delim`) to an int slice.
// Example:
// 		ParseIntSlice("1,2,3", ",") => [1,2,3]
func ParseIntSlice(s, delim string) ([]int, error) {
	slice := []int{}
	for _, token := range strings.Split(s, delim) {
		elem, err := strconv.Atoi(token)
		if err != nil {
			return nil, err
		}
		slice = append(slice, elem)
	}
	return slice, nil
}

// IntSliceToString builds a string (delimited by `delim`) from an int slice.
// Example:
// 		IntSliceToString([1,2,3], ",") => "1,2,3"
func IntSliceToString(ints []int, delim string) string {
	tokens := make([]string, len(ints))
	for i, val := range ints {
		tokens[i] = strconv.Itoa(val)
	}
	return strings.Join(tokens, delim)
}

func WrapErr(msg string, err error) error {
	return errors.New(msg + err.Error())
}

func WrapErrFunc(msg string) func(err error) error {
	return func(err error) error {
		if err == nil {
			return nil
		}
		return WrapErr(msg, err)
	}
}
