package util

import (
	"bytes"
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"time"
)

// NullTime provides a struct that is usable by database/sql implementations
// and handles JSON serialization as you would expect
type NullTime struct {
	Valid bool
	Time  time.Time
}

var _ json.Marshaler = (*NullTime)(nil)
var _ json.Unmarshaler = (*NullTime)(nil)
var _ sql.Scanner = (*NullTime)(nil)
var _ driver.Value = NullTime{}

// MarshalJSON implements the encoding/json.Marshaler interface
func (nt *NullTime) MarshalJSON() ([]byte, error) {
	if nt.Valid {
		return json.Marshal(nt.Time)
	}
	return json.Marshal(nil)
}

// UnmarshalJSON implements the encoding/json.Unmarshaler interface
func (nt *NullTime) UnmarshalJSON(b []byte) error {
	if bytes.Equal(b, nullJSON) {
		nt.Valid = false
		return nil
	}
	err := json.Unmarshal(b, &nt.Time)
	nt.Valid = (err == nil)
	return err
}

// Scan implements the database/sql.Scanner interface
func (nt *NullTime) Scan(value interface{}) error {
	nt.Time, nt.Valid = value.(time.Time)
	return nil
}

// Value implements the database/sql/driver.Value interface
func (nt NullTime) Value() (driver.Value, error) {
	if nt.Valid {
		return nt.Time, nil
	}
	return nil, nil
}

var nullJSON []byte = []byte("null")
