package main

import (
	"bytes"
	"crypto/md5"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"sort"
	"strconv"
	"time"
)

type Mixpanel struct {
	ApiKey  string
	Secret  string
	Format  string
	BaseUrl string
}

type EventQueryResult struct {
	LegendSize int `json:legend_size`
	Data       struct {
		Series []string                  `json:series`
		Values map[string]map[string]int `json:values`
	} `json:data`
}

type ExportQueryResult struct {
	Event      string                 `json:event`
	Properties map[string]interface{} `json:properties`
}

func NewMixpanel(key string, secret string) *Mixpanel {
	m := new(Mixpanel)
	m.Secret = secret
	m.ApiKey = key
	m.Format = "json"
	m.BaseUrl = "http://mixpanel.com/api/2.0"
	return m
}

func (m *Mixpanel) AddExpire(params *map[string]string) {
	if (*params)["expire"] == "" {
		(*params)["expire"] = strconv.FormatInt(ExpireInDays(5), 10)
	}
}

func (m *Mixpanel) AddSig(params *map[string]string) {
	keys := make([]string, 0)

	(*params)["api_key"] = m.ApiKey
	(*params)["format"] = m.Format

	for k, _ := range *params {
		keys = append(keys, k)
	}
	sort.StringSlice(keys).Sort()
	// fmt.Println(s)

	var buffer bytes.Buffer
	for _, key := range keys {
		value := (*params)[key]
		buffer.WriteString(fmt.Sprintf("%s=%s", key, value))
	}
	buffer.WriteString(m.Secret)
	// fmt.Println(buffer.String())

	hash := md5.New()
	hash.Write(buffer.Bytes())
	sigHex := fmt.Sprintf("%x", hash.Sum([]byte{}))
	(*params)["sig"] = sigHex
}

func (m *Mixpanel) makeRequest(action string, params map[string]string) ([]byte, error) {
	m.AddExpire(&params)
	m.AddSig(&params)

	var buffer bytes.Buffer
	for key, value := range params {
		value = url.QueryEscape(value)
		buffer.WriteString(fmt.Sprintf("%s=%s&", key, value))
	}

	uri := fmt.Sprintf("%s/%s?%s", m.BaseUrl, action, buffer.String())
	uri = uri[:len(uri)-1]

	client := new(http.Client)
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		errStr := "Unable to create new request"
		log.Println(errStr, err)
		return []byte{}, errors.New(errStr)
	}

	resp, err := client.Do(req)
	if err != nil {
		errStr := "Unable to create new get response"
		log.Println(errStr, err)
		return []byte{}, errors.New(errStr)
	}
	defer resp.Body.Close()
	bytes, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		errStr := "Unable to create new read body"
		log.Println(errStr, err)
		return []byte{}, errors.New(errStr)
	}
	return bytes, err
}

func ExpireInDays(days int64) int64 {
	return time.Now().Add(time.Duration(int64(time.Hour) * days * 24)).Unix()
}

func ExpireInHours(hours int64) int64 {
	return time.Now().Add(time.Duration(int64(time.Hour) * hours)).Unix()
}
