package multiconfig

import (
	"fmt"
	"testing"
	"time"
)

type TestDurationCase struct {
	name          string
	envVal        string
	configVal     string
	flagVal       string
	defaultVal    time.Duration
	expectedValue time.Duration
	shouldErr     bool
}

func (t *TestDurationCase) testEnv(name string) string {
	return t.envVal
}

var durationTestCases = []TestDurationCase{
	{
		name:          "flagTest",
		envVal:        "50s",
		configVal:     "80s",
		flagVal:       "100s",
		defaultVal:    5 * time.Second,
		expectedValue: 100 * time.Second,
		shouldErr:     false,
	},
	{
		name:          "envTest",
		envVal:        "50s",
		configVal:     "80s",
		expectedValue: 50 * time.Second,
		defaultVal:    5 * time.Second,
		shouldErr:     false,
	},
	{
		name:          "configTest",
		configVal:     "80s",
		expectedValue: 80 * time.Second,
		defaultVal:    5 * time.Second,
		shouldErr:     false,
	},
	{
		name:          "defaultTest",
		expectedValue: 5 * time.Second,
		defaultVal:    5 * time.Second,
		shouldErr:     false,
	},
}

func TestDuration(t *testing.T) {
	for _, c := range durationTestCases {
		var dest time.Duration

		config := NewConfig()
		config.envGetter = c.testEnv
		if c.flagVal != "" {
			config.args = []string{fmt.Sprintf("-%s", c.name), c.flagVal}
		}

		if c.configVal != "" {
			config.config[c.name] = c.configVal
		}

		config.DurationVar(&dest, c.name, c.defaultVal, "the foo var")

		err := config.Parse()
		if err != nil && !c.shouldErr {
			t.Errorf("TestDuration[%s] encountered err: %v", c.name, err)
			continue
		}

		if err == nil && c.shouldErr {
			t.Errorf("TestDuration[%s] should have errored.", c.name)
			continue
		}

		if !c.shouldErr && dest != c.expectedValue {
			t.Errorf("TestDuration[%s] Got %d; expected %d", c.name, dest, c.expectedValue)
		}
	}
}
