/**
 * RedshiftConfig
 * @property {string} host name of redshift cluster
 * @property {string} dbName of database in cluster to connect to
 * @property {string} dbUser of user in cluster to connect with
 * @property {number} [port] of the cluster, default: 5439
 * @property {string} [clusterIdentifier] of the cluster, default: first segment of host URL.
 * @property {string} [unloadRole] attached to redshift cluster that can be used to unload.
 *    If provided, Conductor CDK will create an IAM role that has unload access to the project's
 *    automatically generated S3 bucket, and give this unloadRole assume role permission to the
 *    created role.
 */
export interface RedshiftConfig {
  readonly host: string;
  readonly dbName: string;
  readonly dbUser: string;
  readonly port?: number;
  readonly clusterIdentifier?: string;
  readonly unloadRole?: string;
}

/**
 * VPCConfig
 * @property {string} vpc ID to provision resources in
 * @property {Array<string>} subnets to provision resources in
 * @property {Array<string>} securityGroups to attach to provisioned resources
 */
export interface VPCConfig {
  readonly vpc: string;
  readonly subnets: Array<string>;
  readonly securityGroups: Array<string>;
}

/**
 * AirflowConfig
 * @property {string} [dagsS3Path] is the s3 bucket to upload DAG files to.
 * @property {string} [mwaaEnvironmentName] of the AWS Managed Workflows for Apache Airflow (MWAA) environment.
 */
export interface AirflowConfig {
  readonly dagsS3Path?: string;
  readonly mwaaEnvironmentName?: string;
}

/**
 * SageMakerConfig
 * @property {string} [executionRole] to use if provided, otherwise an auto-generate role will be used
 */
export interface SageMakerConfig {
  readonly executionRole?: string;
}

/**
 * PagerdutyConfig
 * @property {string} connId in Airflow to use, defaults to "pagerduty".
 *    See https://docs.aws.amazon.com/mwaa/latest/userguide/connections-secrets-manager.html for setup
 *    instructions in MWAA.
 */
export interface PagerdutyConfig {
  readonly connId: string;
}

/**
 * ProjectBucketConfig
 * @property {boolean} airflowExecutionRoleAccess Gives read/write access to the Airflow execution role.
 * @property {string} readAccessRoles List of Role ARNs that have read access on the project bucket.
 */
export interface ProjectBucketConfig {
  readonly airflowExecutionRoleAccess: boolean;
  readonly readAccessRoles: Array<string>;
}

/**
 * EnvironmentConfig
 * @property {string} accountId of the AWS account resources live in
 * @property {accountName} accountName of the AWS account, used to fetch deploy credentials locally
 *    This should be standardize within your team to avoid naming discrepancies during development.
 * @property {string} defaultRegion to deploy resources to
 * @property {MWAAConfig | AirflowConfig} airflow config, which determines where DAGs should run
 * @property {RedshiftConfig} [redshift]
 * @property {VPCConfig} [vpc]
 * @property {SageMakerConfig} [sagemaker]
 * @property {PagerdutyConfig} [pagerduty]
 * @property {boolean} [prod] Whether or not this environment is production, in which case user
 *    confirmation wil be required to run Conductor CLI commands, and a Pagerduty alarm on DAG
 *    failure will be provisioned by default.
 */
export interface EnvironmentConfig {
  readonly accountId: string;
  readonly accountName: string;
  readonly defaultRegion: string;
  readonly airflow: AirflowConfig;
  // Optional resource config
  readonly projectBucket?: ProjectBucketConfig;
  readonly redshift?: RedshiftConfig;
  readonly vpc?: VPCConfig;
  readonly sagemaker?: SageMakerConfig;
  readonly pagerduty?: PagerdutyConfig;
  readonly prod?: boolean;
}
