---
title: "Config"
page-category: "searchable"
---

The page documents configuration types for Conductor.

## Project
Conductor configuration is organized around a `Project` object.
A project organizes a group of related workflows that together solve some problem.

An example project might be: build a spam detection model, composed of the following workflows:
- DAG to ingest and preprocess data hourly.
- DAG to train, validate, and deploy the model from the preprocessed data daily.

Project configuration is imported from `conductor.config`, and Conductor expects exactly one instance of the `Project` class named `project` to be defined in the `project_config.py` file at the root of your repository.

```python
from conductor.config import AirflowConfig, EnvironmentConfig, Project

staging_airflow_config = AirflowConfig(mwaa_environment_name="my-mwaa-environment-name")
staging_env = EnvironmentConfig(
    account_id="123456789",
    account_name="my-account-name",
    default_region="us-west-2",
    airflow=staging_airflow_config,
)

# Conductor expects the below project variable in the project_config.py file.
project = Project(name="my-project", environments={"staging": staging_env})
```

The `Project` class is instantiated with two keyword args: `name`, the name of your project, and `environments`, a dict from the name of an AWS environment to the corresponding `EnvironmentConfig` object.
The same `EnvironmentConfig` object may be used twice, for example:

```python
project = Project(name="my-project", environments={"dev", staging_env, "staging": staging_env})
```

The above uses the staging environment for both staging and development.
Resource names typically include the environment name to avoid collision in this use case.

The following sections detail the environment configuration options available for Conductor projects.

## Environment
All args below are passed as keyword arguments for clarity.

#### EnvironmentConfig
**Args**
- `account_id: str` Account ID of the AWS account resources live in
- `account_name: str` Name of the AWS account, used to fetch deploy credentials locally.
This should be standardize within your team to avoid naming discrepancies during development.
- `default_region: str` Default region to deploy resources to.
- `airflow: AirflowConfig` Airflow config, which determines where DAGs should run.

**Optional Args**
- `redshift: RedshiftConfig` Redshift configuration.
- `vpc: VPCConfig` VPC configuration.
- `sagemaker: SageMakerConfig` SageMaker configuration.
- `prod: boolean` Whether or not this environment is production, in which case user confirmation wil be required to run Conductor CLI commands.

#### AirflowConfig
**Optional Args**
- `dags_s3_path: str` The s3 bucket to upload DAG files to.
- `mwaa_environment_name: str` Name of the AWS Managed Workflows for Apache Airflow (MWAA) environment.

#### RedshiftConfig
**Args**
- `host: str` Host name of redshift cluster
- `db_name: str` Name of database in cluster to connect to
- `db_user: str` User in cluster to connect with

**Optional Args**
- `port: int` Port on the cluster host to connect to, default: 5439
- `cluster_identifier: str` Identifier of the Redshift cluster, default: first segment of host URL.
- `unload_role: str` IAM role attached to redshift cluster that can be used to unload.
If provided, Conductor CDK will create an IAM role that has unload access to the project's automatically generated S3 bucket, and give this unloadRole assume role permission to the created role.

#### VPCConfig
**Args**
- `vpc: str` ID of the VPC to provision resources in.
- `subnets: List[str]` Subnets to provision resources in.
- `security_groups: List[str]` Security groups to attach to provisioned resources.

#### SageMakerConfig
**Optional Args**
- `execution_role: str` Custom execution role to use, if provided. 
If not, an auto-generate role will be used.

#### PagerdutyConfig
**Args**
- `conn_id: str` Connection ID that contains the Pagerduty integration key.
If your Secrets Manager key is stored as `airflow/connection/my-pagerduty-key` then the value of `conn_id` should be `my-pagerduty-key`.
The connection itself should be of the form `none://:pagerduty@?routing_key=<your integration key>`.


## Code as Configuration
Conductor subscribes to the **code as configuration** philosophy, so that configs can be statically verified and distributed as code artifacts.
For example, by publishing your team's environment config as a pypi package, future `project_config.py` files simply become:

`poetry add my-team-environments`

```python
# project_config.py
from conductor.config import Project
from my_team_environments import environments

project = Project(name="my-project", environments=environments)
```
This has the added benefit of making migrations to new team infrastructure very easy: simply upgrade the version of the team environments package!




