<?php
// äüö UTF8-FTW
class RedisSessionHandler implements SessionHandlerInterface
{
	public $ttl = 1800; // 30 minutes default
	protected $redis;
	protected $prefix;

	/**
	 * @var integer Default PHP max execution time in seconds
	 */
	const DEFAULT_MAX_EXECUTION_TIME = 300;

	/**
	 * @var boolean Indicates an sessions should be locked
	 */
	private $locking;

	/**
	 * @var boolean Indicates an active session lock
	 */
	private $locked;

	/**
	 * @var string Session lock key
	 */
	private $lockKey;

	/**
	 * @var integer Microseconds to wait between acquire lock tries
	 */
	private $spinLockWait;

	/**
	 * @var integer Maximum amount of seconds to wait for the lock
	 */
	private $lockMaxWait;

	/**
	 * @var boolean Readonly mode, no locking, no writing
	 */
	private $readOnly;


	public function __construct($redis = null, $prefix = 'PHPSESSID:', $locking = true, $spinLockWait = 50000, $read_only = false)
	{
		if($redis === null)
		{
			$redis = new Redis();
			if($redis->connect('better-desk.8rs2gm.0001.usw1.cache.amazonaws.com', 6379, 5.0) === false)
			{
				header('HTTP/1.1 503 Service Unavailable', true, 503);
				echo '503 Service Unavailable';
				exit();
			}
		}
		$this->redis = $redis;
		
		if($prefix === null)
		{
			$prefix = 'PHPSESSID:';
		}
		$this->prefix = $prefix;
		
		if($locking === null)
		{
			$locking = true;
		}
		$this->locking = $locking;
		$this->locked = false;
		$this->lockKey = null;
		
		if($spinLockWait === null)
		{
			$spinLockWait = 50000;
		}
		$this->spinLockWait = $spinLockWait;
		$this->lockMaxWait = ini_get('max_execution_time');
		if(!$this->lockMaxWait)
		{
			$this->lockMaxWait = self::DEFAULT_MAX_EXECUTION_TIME;
		}
		
		if($read_only === null)
		{
			$read_only = false;
		}
		$this->readOnly = $read_only;
	}

	public function open($savePath, $sessionName)
	{
		// No action necessary because connection is injected
		// in constructor and arguments are not applicable.
		return true;
	}

	public function close()
	{
		if($this->locking === true && $this->locked === true && $this->readOnly === false)
		{
			$this->unlockSession();
		}
		$this->redis->close();
		$this->redis = null;
		unset($this->redis);
		return true;
	}

	public function read($id)
	{
		$id = $this->prefix . $id;
		if($this->locking === true && $this->locked === false && $this->readOnly === false)
		{
			if($this->lockSession($id) === false)
			{
				return false;
			}
		}
		try {
			$sessData = $this->redis->get($id);
			$this->redis->expire($id, $this->ttl);
		} catch(RedisException $E) {
			// print_r($E);
			header('HTTP/1.1 503 Service Unavailable', true, 503);
			echo '503 Service Unavailable';
			exit();
		}
		return $sessData;
	}

	public function write($id, $data)
	{
		$sessionId = $this->prefix . $id;
		if($this->readOnly === false)
		{
			try {
				$this->redis->set($sessionId, $data, $this->ttl);
			} catch(RedisException $E) {
				// print_r($E);
			}
		}
		return true;
	}

	public function destroy($id)
	{
		if($this->readOnly === false)
		{
			try {
				$this->redis->del($this->prefix . $id);
			} catch(RedisException $E) {
				// print_r($E);
			}
		}
		return true;
	}

	public function gc($maxLifetime)
	{
		// no action necessary because we are using EXPIRE
		return true;
	}
	
	private function lockSession($sessionId)
	{
		if($this->readOnly === false)
		{
			$this->lockKey = $sessionId.'.lock';
			$try_lock_until_time = time() + $this->lockMaxWait;
			
			while(time() < $try_lock_until_time)
			{
				$last_lock_try_time = microtime(true);
				try {
					$success = $this->redis->set($this->lockKey, '1', array('nx', 'ex' => $this->lockMaxWait + 1));
				} catch(RedisException $E) {
					// print_r($E);
					header('HTTP/1.1 503 Service Unavailable', true, 503);
					echo '503 Service Unavailable';
					exit();
				}
				if($success === true)
				{
					$this->locked = true;
					return true;
				}
				$sleep_time = $this->spinLockWait - round((microtime(true) - $last_lock_try_time) * 1000000);
				if($sleep_time > 0)
				{
					usleep($this->spinLockWait);
				}
			}

			return false;
		}
		else
		{
			return true;
		}
	}

	private function unlockSession()
	{
		if(!empty($this->lockKey) && $this->readOnly === false)
		{
			try {
				$this->redis->del($this->lockKey);
			} catch(RedisException $E) {
				// print_r($E);
			}
		}
		$this->locked = false;
	}
}

?>