// ÜÖÄ UTF8-FTW
package main

import (
		"bufio"
		"fmt"
		"os"
		// "net"
		"regexp"
		"crypto/tls"
		"strings"
		"log"
		"encoding/json"
		"github.com/oleiade/lane"
		"time"
		"strconv"
		"database/sql"
		_ "github.com/go-sql-driver/mysql"
)

type Message struct {
	Command string
	Room string
	Room_id string
	Nick string
	Nick_id string
	Target string
	Body string
	Tags string
	Tags_int int64
	Time int64
}

const CHATLOG_PREFIX string = "chatlog_"

// Chat message queue, 1 entry per chat message
var chatfeed_message_queue = lane.NewDeque()
// Mysql insert queue, 1 entry per query
var mysql_insert_queue = lane.NewDeque()
var msg_since_last_check uint = 0

func main() {
	// Start firehose https thread
	go chat_feed_thread()
	// Start mysql create query thread (converts chatfeed_message_queue entries to mysql_insert_queue entries)
	go mysql_create_query_thread()
	// Start mysql insert thread (inserts mysql_insert_queue into the DB)
	go mysql_insert_query_thread()

	// Display current status info every sec
	var chatfeed_zero_message_check int = 0
	for {
		time.Sleep(1 * time.Second)
		fmt.Println(time.Now().Format("2006-01-02 15:04:05 -07:00 MST"), "=> Message Queue:", chatfeed_message_queue.Size(), "|| Mysql Queue: ", mysql_insert_queue.Size(), "|| MSG/s:", msg_since_last_check)
		if msg_since_last_check == 0 {
			chatfeed_zero_message_check = chatfeed_zero_message_check + 1
		} else {
			chatfeed_zero_message_check = 0
		}
		msg_since_last_check = 0

		if chatfeed_zero_message_check >= 15 && mysql_insert_queue.Size() == 0 {
			os.Exit(0)
		}
	}
}

// Escapes problematic chars so we are safe from SQL injection
func mysql_real_escape_string(s string) string {
	// return str_replace(array('\\', "\0", "\n", "\r", "'", '"', "\x1a"), array('\\\\', '\\0', '\\n', '\\r', "\\'", '\\"', '\\Z'), $inp);
	s = strings.Replace(s, "\\", "\\\\", -1)
	s = strings.Replace(s, "\x00", "\\x00", -1)
	s = strings.Replace(s, "\n", "\\n", -1)
	s = strings.Replace(s, "\r", "\\r", -1)
	s = strings.Replace(s, "'", "\\'", -1)
	s = strings.Replace(s, `"`, `\"`, -1)
	s = strings.Replace(s, "\x1a", "\\Z", -1)
	return s
}

func parse_tags_string(tags_string string, wanted_tag string) string {
	tags_array := strings.Split(tags_string, ";")
	for _, tag := range tags_array {
		tag_parsed := strings.SplitN(tag, "=", 2)
		if tag_parsed[0] == wanted_tag {
			return tag_parsed[1]
		}
	}

	return ""
}

func mysql_insert_query_thread() {
	var mysql_query_string string
	// Create regex to find the date suffix for the chatlog table
	r, _ := regexp.Compile("\\." + CHATLOG_PREFIX + "(20[123][0-9]_[01][0-9]_[0-3][0-9])'")
	// Open the SQL connection
	db, err := sql.Open("mysql", "chatlogs:F8xB9NDMSw1KTldF@tcp(leviathan-chatlogs-mysql-prod.cjmd1imzurdd.us-west-2.rds.amazonaws.com:3306)/chatlogs?charset=utf8mb4,utf8")
	if err != nil {
		panic(err.Error())
	}
	defer db.Close()

	// Make sure the SQL connection works
	err = db.Ping()
	if err != nil {
		panic(err.Error())
	}

	for {
		query_string := mysql_insert_queue.Shift()
		if query_string != nil {
			mysql_query_string = query_string.(string)
			// fmt.Println(mysql_query_string)
			_, err := db.Exec(mysql_query_string)
			if err != nil {
				// If it's a table doesn't exist error
				if len(err.Error()) >= 12 && err.Error()[0:11] == "Error 1146:" {
					table_suffix := r.FindStringSubmatch(err.Error())
					if table_suffix != nil {
						// Re add the query to the start of the queue
						mysql_insert_queue.Prepend(query_string)
						// Create new table with this date
						create_new_chatlog_table(table_suffix[1])
					} else {
						log.Fatal(err)
					}
				} else {
					log.Fatal(err)
				}
			}
			// fmt.Println(mysql_query_string)
		} else {
			// Nothing to insert so wait
			time.Sleep(50 * time.Millisecond)
		}
	}
}

func create_new_chatlog_table(date_string string) {
	query := "CREATE TABLE IF NOT EXISTS `" + CHATLOG_PREFIX + date_string + "` (" +
		"`ID` int(10) unsigned NOT NULL AUTO_INCREMENT," +
		"`room` varchar(254) CHARACTER SET utf8mb4 NOT NULL," +
		"`room_id` int(10) unsigned NULL DEFAULT NULL," +
		"`nick` varchar(254) CHARACTER SET utf8mb4 NOT NULL," +
		"`nick_id` int(10) unsigned NULL DEFAULT NULL," +
		"`message` varchar(4096) CHARACTER SET utf8mb4 NOT NULL," +
		"`tags_int` tinyint(3) unsigned NOT NULL DEFAULT '0' COMMENT 'None = 0, Mod = 1, global_mod = 2, admin = 4, staff = 8, turbo = 16, subscriber = 32, cheer_message = 64, CLEARCHAT = 128'," +
		"`timestamp` timestamp NULL DEFAULT NULL," +
		"PRIMARY KEY (`ID`)," +
		"KEY `nick` (`nick`(25))," +
		"KEY `nick_id` (`nick_id`)," +
		"KEY `room` (`room`(26))," +
		"KEY `room_id` (`room_id`)" +
	") AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb4 ROW_FORMAT=COMPRESSED"
	// Prepend the query to mysql queue so it gets executed before any other chatlog insert queries
	mysql_insert_queue.Prepend(query)
}

func current_pt_date_string(unixtime int64) string {
	t := time.Unix(unixtime, 0)
	pt, _ := time.LoadLocation("America/Los_Angeles")
	// YYYY_MM_DD
	return t.In(pt).Format("2006_01_02")
}

func mysql_create_query_thread() {
	// Max chat log lines per single SQL query
	var max_entries_at_once int = 1000
	// Max time between SQL queries even if we haven't hit the MAX above (in Seconds)
	var max_queue_wait_time int64 = 10
	var mysql_query_string string
	var last_query_time int64 = time.Now().Unix()

	for {
		if chatfeed_message_queue.Size() > 0 && (chatfeed_message_queue.Size() >= max_entries_at_once || (last_query_time < (time.Now().Unix() - max_queue_wait_time))) {
			var parts []string
			// Get the top message date for the table suffix
			top_single_message := chatfeed_message_queue.First()
			top_single_message_struct := top_single_message.(Message)
			table_suffix := current_pt_date_string(top_single_message_struct.Time)

			parts = append(parts, "INSERT INTO `", CHATLOG_PREFIX, table_suffix, "` (`room`, `room_id`, `nick`, `nick_id`, `message`, `tags_int`, `timestamp`) VALUES ")
			for i := 0; i < max_entries_at_once; i++ {
				single_message := chatfeed_message_queue.Shift()
				// Only do stuff if it exists and if it's from the same day
				if single_message != nil {
					single_message_struct := single_message.(Message)
					// Only add entries from the say day to the same query (else they would go into the wrong table)
					if table_suffix == current_pt_date_string(single_message_struct.Time) {
						// If it's not the first one we should prepend it with ,
						if i > 0 {
							parts = append(parts, ",")
						}
						parts = append(parts, "('", mysql_real_escape_string(single_message_struct.Room), "',")
						if single_message_struct.Room_id == "" {
							parts = append(parts, "NULL,")
						} else {
							parts = append(parts, "'", mysql_real_escape_string(single_message_struct.Room_id), "',")
						}
						parts = append(parts, "'", mysql_real_escape_string(single_message_struct.Nick), "',")
						if single_message_struct.Nick_id == "" {
							parts = append(parts, "NULL,")
						} else {
							parts = append(parts, "'", mysql_real_escape_string(single_message_struct.Nick_id), "',")
						}
						parts = append(parts, "'", mysql_real_escape_string(single_message_struct.Body), "','", strconv.FormatInt(single_message_struct.Tags_int, 10), "',FROM_UNIXTIME('", strconv.FormatInt(single_message_struct.Time, 10), "'))")
					} else {
						// Readd the message at the start of the queue and break out of this.
						chatfeed_message_queue.Prepend(single_message)
						break
					}
				} else {
					break
				}
			}
			mysql_query_string = strings.Join(parts, "")
			mysql_insert_queue.Append(mysql_query_string)
			last_query_time = time.Now().Unix()
		} else {
			// If nothing is enough yet just wait
			time.Sleep(50 * time.Millisecond)
		}
	}
}

func chat_feed_thread() {
	var tags_int int64 = 0
	check_next_line := false
	mod_found := false

	tls_conf := &tls.Config{}

	// Connect part
	chat_feed_start:
	conn, err := tls.Dial("tcp", "firehose.internal.twitch.tv:443", tls_conf)
	if err != nil {
		// handle error
		log.Fatal(err)
	}
	defer conn.Close()
	fmt.Fprintf(conn, "GET /firehose HTTP/1.1\r\nHost: firehose.internal.twitch.tv\r\n\r\n")


	// Scanner part
	scanner := bufio.NewScanner(conn)
	scanner.Split(bufio.ScanLines)
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		// fmt.Println(line)
		if len(line) > 0 {
			if line == "event: privmsg" {
				check_next_line = true
			} else if check_next_line == true {
				if len(line) >= 7 && line[0:6] == "data: " {
					json_string := line[6:len(line)]
					b := []byte(json_string)
					var m Message
					json_err := json.Unmarshal(b, &m)
					if json_err != nil {
						fmt.Println(scanner.Err())
						fmt.Println(json_err)
						fmt.Println(line)
					} else {
						// Only handle normal messages (no timeouts or other stuff which gets sent via the firehose)
						if m.Command == "" {
							m.Time = time.Now().Unix()
							// Parse the tags into bytewise int
							tags_int = 0
							if len(m.Tags) > 0 {
								mod_found = false
								// Mod = 1
								if strings.Index(m.Tags, "mod=1") != -1 {
									tags_int = tags_int + 1;
									mod_found = true;
								}
								// Mod = 1
								if mod_found == false && strings.Index(m.Tags, "user-type=mod") != -1 {
									tags_int = tags_int + 1
								}
								// global_mod = 2
								if strings.Index(m.Tags, "user-type=global_mod") != -1 {
									tags_int = tags_int + 2
								}
								// admin = 4
								if strings.Index(m.Tags, "user-type=admin") != -1 {
									tags_int = tags_int + 4
								}
								// staff = 8
								if strings.Index(m.Tags, "user-type=staff") != -1 {
									tags_int = tags_int + 8
								}
								// turbo = 16
								if strings.Index(m.Tags, "turbo=1") != -1 {
									tags_int = tags_int + 16
								}
								// subscriber = 32
								if strings.Index(m.Tags, "subscriber=1") != -1 {
									tags_int = tags_int + 32
								}
								// cheer_message = 64
								if strings.Index(m.Tags, "bits=") != -1 {
									tags_int = tags_int + 64
								}
							}
							m.Tags_int = tags_int

							// Parse the room-id
							m.Room_id = parse_tags_string(m.Tags, "room-id")

							// Parse the user-id
							m.Nick_id = parse_tags_string(m.Tags, "user-id")

							// Add it to the queue
							chatfeed_message_queue.Append(m)
							msg_since_last_check++
							// fmt.Println(m)
						} else if m.Command == "CLEARCHAT" {
							// Log timeout / bans

							// We need to create a new struct to get the format right
							var clearchat_msg Message
							clearchat_msg.Time = time.Now().Unix()
							clearchat_msg.Tags_int = 128 // 128 for CLEARCHAT
							clearchat_msg.Room = m.Target
							clearchat_msg.Nick = m.Body
							clearchat_msg.Body = m.Tags

							// Add it to the queue
							chatfeed_message_queue.Append(clearchat_msg)
							msg_since_last_check++
							// fmt.Println(clearchat_msg)
						}
					}
				}
				check_next_line = false
			}
		}
	}

	// Restart the chat feed if we get here (probably due some kind of dc)
	goto chat_feed_start
}
