class FileUploadsController < ApplicationController
  require 'aws-sdk'
  require 'base64'
  require 'net/http'
  require 'rmagick'

  def index
    @file_uploads = FileUpload.order(id: :desc).paginate(page: params[:page])
  end

  def new
    @file_upload = FileUpload.new
  end

  def create
    audit = {}
    @file_upload = FileUpload.new(file_uploads_params)
    # Set the user_id to the current_user ID
    @file_upload.user_id = current_user.id
    # Remove trailing whitespaces
    @file_upload.channel = @file_upload.channel.strip
    # Downcase the channel
    @file_upload.channel = @file_upload.channel.downcase

    # Check if the channel name is valid
    unless /\A[0-9a-zA-z_]{1,}\z/.match(@file_upload.channel)
      flash.now[:danger] = "This is not a valid channel name."
      render 'new' and return
    end

    # Set the redirect cookie based on the post request
    unless request.format.symbol.to_s == "json"
      if params[:redirect_after_file_upload] && params[:redirect_after_file_upload] == "true"
        cookies.permanent[:redirect_after_file_upload] = "true"
      else
        cookies.permanent[:redirect_after_file_upload] = "false"
      end
    end

    if params[:image_paste].blank? == false
      audit[:type] = "paste"
      # Get the raw file content from the base64 encoded form string
      file_body = Base64.decode64(params[:image_paste].split(",", 2)[1])
      # Get the extension
      file_extension = ((params[:image_paste].split(",", 2)[0]).split("/", 2)[1]).split(";", 2)[0]

      # Check if there is a paste
      if file_body.blank?
        flash.now[:danger] = "The paste was empty"
        render 'new' and return
      end

      # Check if the extension is allowed
      unless is_valid_extension?(file_extension)
        flash.now[:danger] = "This file extension is not allowed to be uploaded."
        render 'new' and return
      end

      new_file_name = "file-uploads/" + generate_random_string(16) + "/" + create_filename(@file_upload.channel, "." + file_extension)

      s3_obj = s3_obj(new_file_name)
      s3_obj.put(body: file_body, acl: 'public-read', content_type: content_type_based_on_ext(file_extension))

      @file_upload.file = File.basename(new_file_name)
      @file_upload.url = s3_obj.public_url

    elsif params[:url].blank? == false
      audit[:type] = "url"
      audit[:url] = params[:url]

      # check if the url is valid
      unless params[:url] =~ /\A#{URI::regexp(['http', 'https'])}\z/
        flash.now[:danger] = "The provided URL \"#{params[:url]}\" is not valid."
        render 'new' and return
      end

      # check if the url is a whitelisted host
      allowed_hosts = %w(www-origin.twitch.tv www.twitch.tv twitch.tv static-cdn.jtvnw.net static.justin.tv web-cdn.ttvnw.net)
      url = URI.parse(params[:url])
      unless allowed_hosts.include?(url.host)
        flash.now[:danger] = "This URL is not whitelisted. Due to NDA restrictions you shouldn't use 3rd party sites to upload your screenshots anyway."
        render 'new' and return
      end

      begin
          # Download the file
          http = Net::HTTP.new(url.host, url.port)
          http.use_ssl = true if url.scheme == "https"
          http.open_timeout = 3
          http.read_timeout = 15
          req = Net::HTTP::Get.new(url.request_uri)
          res = http.request(req)
          unless res.is_a?(Net::HTTPSuccess)
            # Just give a 400 error in case it was a Ajax request
            if request.format.symbol.to_s == "json"
              render(json: {status: 400, error: true, error_message: "There is no image at this url"}, :status => 400) and return
            end

            flash.now[:danger] = "There was an error downloading the file located at: #{params[:url]}"
            render 'new' and return
          end
        rescue Exception => e
          # Just give a 400 error in case it was a Ajax request
          if request.format.symbol.to_s == "json"
            render(json: {status: 400, error: true, error_message: "There is no image at this url"}, :status => 400) and return
          end

          flash.now[:danger] = "There was an issue downloading the file from the provided URL."
          render 'new' and return
      end

      # Check if there is a file body
      if res.body.blank?
        flash.now[:danger] = "The file at the URL has no content"
        render 'new' and return
      end

      # Get the file extension
      if res['Content-Type']
        file_extension = res['Content-Type'].split("/", 2)[1]
      else
        # Try to get the extension from the url
        file_extension = File.extname(File.basename(params[:url])).remove(".")
      end

      # Check if we got an extension
      if file_extension.blank?
        flash.now[:danger] = "Could not determine the file extension for this URL"
        render 'new' and return
      end

      # Check if the extension is allowed
      unless is_valid_extension?(file_extension)
        flash.now[:danger] = "This file extension is not allowed to be uploaded."
        render 'new' and return
      end

      file_body = res.body
      # Check if this is a channel image url
      if is_channel_image_url?(params[:url])
        @channel_image_url_type = channel_image_url_type(params[:url])
        audit[:channel_image] = true
        audit[:channel_image_type] = @channel_image_url_type

        # Add additional info before uploading it
        file_body = add_info_to_channel_image(file_body, params[:url], @channel_image_url_type)
      end

      new_file_name = "file-uploads/" + generate_random_string(16) + "/" + create_filename(@file_upload.channel, "." + file_extension)

      s3_obj = s3_obj(new_file_name)
      s3_obj.put(body: file_body, acl: 'public-read', content_type: content_type_based_on_ext(file_extension))

      @file_upload.file = File.basename(new_file_name)
      @file_upload.url = s3_obj.public_url

    elsif params[:file].blank? == false
      audit[:type] = "file"
      new_file_name = "file-uploads/" + generate_random_string(16) + "/" + create_filename(@file_upload.channel, File.extname(params[:file].original_filename))

      # Get file extension
      file_extension = File.extname(params[:file].original_filename).remove(".")

      # Check if the extension is allowed
      unless is_valid_extension?(file_extension)
        flash.now[:danger] = "This file extension is not allowed to be uploaded."
        render 'new' and return
      end

      s3_obj = s3_obj(new_file_name)
      s3_obj.upload_file(params[:file].tempfile, acl: 'public-read', content_type: content_type_based_on_ext(file_extension))

      @file_upload.file = File.basename(new_file_name)
      @file_upload.url = s3_obj.public_url
    else
      flash.now[:danger] = "You need to either provide a paste, file or URL."
      render 'new' and return
    end

    # Check if we have an upload url aka if the file went to S3
    if @file_upload.url.blank?
      flash.now[:danger] = "There was an error uploading the file to S3."
      render 'new' and return
    end

    # Check if the target is a partner or not
    begin
        channel = ::TwitchUsersService.get_user_by_username(@file_upload.channel)
        @channel_is_partner = ::TwitchPartnershipsService.get_partner_by_id(channel["id"]) unless channel.nil?
      rescue Exception => e
        @channel_is_partner = nil
    end

    if @file_upload.save
      flash.now[:success] = "File successfully uploaded!"

      audit[:channel] = @file_upload.channel
      audit[:channel_is_partner] = @channel_is_partner
      audit[:file_url] = @file_upload.url
      audit[:id] = @file_upload.id
      # Add audit entry
      FileUploadAudit.new(action: "upload", action_by: current_user.id, content: audit.to_json, remote_ip: request.remote_ip).save
    else
      # Error
      render 'new'
    end

    # Redirect if the cookie is set to do so and it's not a Ajax request, else just render normally
    if request.format.symbol.to_s != "json" && cookies[:redirect_after_file_upload] && cookies[:redirect_after_file_upload] == "true"
      if @channel_is_partner && @channel_is_partner == true
        redirect_to("https://leviathan.moderation.twitch.tv/reports/partnerconduct_report?reported_user=#{URI::escape(@file_upload.channel)}&description=#{URI::escape(convert_S3_to_nginx_url(@file_upload.url))}", status: :found)
      else
        redirect_to("#{Settings.twitch_base_url}#{URI::escape(@file_upload.channel)}/report_form?tos_ban=true&description=#{URI::escape(convert_S3_to_nginx_url(@file_upload.url))}", status: :found)
      end
      return
    end
  end

  def destroy
    if current_user && current_user.has_permission_to?(:delete_file_uploads)
      audit = {}
      file_upload = FileUpload.find_by(id: params[:id])
      if file_upload
        audit[:id] = file_upload.id
        audit[:file_url] = file_upload.url
        uri = URI(file_upload.url)

        if uri.path.from(1).blank?
          flash[:danger] = "There was an issue deleting the file"
          redirect_to(file_uploads_path) and return
        end

        s3_client = Aws::S3::Client.new(
          credentials: Aws::Credentials.new(ENV['S3_ACCESS_KEY'], ENV['S3_SECRET_KEY']),
          region: ENV['S3_REGION']
        )

        response = s3_client.delete_object({
            bucket: ENV['S3_BUCKET'],
            key: uri.path.from(1) # from(1) to remove the first /
        })

        # Check if it's deleted
        begin
          s3_obj_status = s3_obj(uri.path.from(1)).exists?
        rescue Aws::S3::Errors::Forbidden # As all of our files are public forbidden means it doesn't exist
          s3_obj_status = false
        end

        # Delete the entry from the DB if the file was deleted from S3
        unless s3_obj_status
          file_upload.destroy!
          flash[:success] = "The file was successfully deleted."

          # Add audit entry
          FileUploadAudit.new(action: "delete", action_by: current_user.id, content: audit.to_json, remote_ip: request.remote_ip).save
        else
          flash[:danger] = "There was an error deleting the file from S3."
        end
      else
        flash[:danger] = "The file you want to delete doesn't exist."
      end
    else
      flash[:danger] = "You don't have permissions to delete files."
    end
    redirect_to(file_uploads_path)
  end

  def audits
    if current_user && current_user.has_permission_to?(:view_file_uploads_audits)
      @file_upload_audits = FileUploadAudit.order(id: :desc).paginate(page: params[:page])
    else
      render 'shared/_no_permission' and return
    end
  end

  def search
    conditions = file_uploads_search_params.delete_if {|k,v| v.blank? }
    if conditions[:file_uploads]
      conditions[:file_uploads] = conditions[:file_uploads].delete_if {|k,v| v.blank? }
    end

    if conditions[:file_uploads] && conditions[:file_uploads].any?
      @file_uploads = FileUpload.where(conditions).order(id: :desc).paginate(page: params[:page])
    end
  end


  private

    def file_uploads_params
      params.require(:file_upload).permit(:channel)
    end

    def file_uploads_search_params
      params.permit(
        file_uploads: [
          :user_id,
          :url,
          :file,
          :channel
        ]
      ).to_h
    end

    def create_filename(channel, extension)
      date = Date.current
      time_string = date.in_time_zone("America/Los_Angeles").strftime("%Y-%m-%d")
      filename = "#{channel}--#{time_string}#{extension}"
    end

    def generate_random_string(number)
      charset = Array('A'..'Z') + Array('a'..'z') + Array('0'..'9')
      Array.new(number) { charset.sample }.join
    end

    def is_valid_extension?(extension)
      allowed_extension = %w(jpg jpeg png gif)
      if allowed_extension.include?(extension)
        return true
      else
        return false
      end
    end

    def s3_obj(filename)
      s3 = Aws::S3::Resource.new(
        credentials: Aws::Credentials.new(ENV['S3_ACCESS_KEY'], ENV['S3_SECRET_KEY']),
        region: ENV['S3_REGION']
      )
      s3_obj = s3.bucket(ENV['S3_BUCKET']).object(filename)
    end

    def is_channel_image_url?(url)
      if profile_image_regexp.match(url.downcase) || profile_banner_regexp.match(url.downcase) || offline_image_regexp.match(url.downcase)
        return true
      else
        return false
      end
    end

    def channel_image_url_type(url)
      if profile_image_regexp.match(url.downcase)
        return "profile_image"
      elsif profile_banner_regexp.match(url.downcase)
        return "profile_banner"
      elsif offline_image_regexp.match(url.downcase)
        return "offline_image"
      else
        return false
      end
    end

    def add_info_to_channel_image(file_body, url, image_type_string)
      if image_type_string == "profile_image"
        type_of_image_string = "Profile image:"
      elsif image_type_string == "profile_banner"
        type_of_image_string = "Profile banner:"
      elsif image_type_string == "offline_image"
        type_of_image_string = "Offline image:"
      end

      channel = channel_image_url_channel(url)
      source = Magick::Image.from_blob(file_body).first

      # Sanity check
      unless source.columns > 0 && source.rows > 0
        return file_body
      end

      new_size_x = source.columns + 20
      if new_size_x < 360
        # We need this size if we have 25 char usernames
        new_size_x = 360
      end
      new_size_y = source.rows + 60


      new_image = Magick::Image.new(new_size_x, new_size_y) { self.background_color = "black" }
      new_image.format = source.format

      txt = Magick::Draw.new
      new_image.annotate(txt, 0, 0, 5, 18, "User:") {
        txt.pointsize = 17
        txt.fill = "#D2D2D2"
      }

      new_image.annotate(txt, 0, 0, 52, 18, channel) {
        txt.pointsize = 16
        txt.fill = "#B4B4B4"
      }

      new_image.annotate(txt, 0, 0, 5, 42, type_of_image_string) {
        txt.pointsize = 17
        txt.fill = "#D2D2D2"
      }

      # Copy original image onto the new one
      new_image.composite!(source, (10 + ((new_size_x - (source.columns + 20)) / 2)).round, 50, Magick::CopyCompositeOp)

      new_image.to_blob
    end

    def channel_image_url_channel(url)
      /\Ahttp[s]?:\/\/(static\-cdn\.jtvnw\.net)\/jtv_user_pictures\/([a-z0-9_]+)\-.*\z/.match(url.downcase).to_a[2]
    end

    def profile_image_regexp
      /\Ahttp[s]?:\/\/(static\-cdn\.jtvnw\.net)\/jtv_user_pictures\/([a-z0-9_]+)\-profile_image\-[a-z0-9]+\-600x600\.[a-z]+\z/
    end

    def profile_banner_regexp
      /\Ahttp[s]?:\/\/(static\-cdn\.jtvnw\.net)\/jtv_user_pictures\/([a-z0-9_]+)\-profile_banner\-[a-z0-9]+\-480\.[a-z]+\z/
    end

    def offline_image_regexp
      /\Ahttp[s]?:\/\/(static\-cdn\.jtvnw\.net)\/jtv_user_pictures\/([a-z0-9_]+)\-channel_offline_image\-[a-z0-9]+\-[0-9]+x[0-9]+\.[a-z]+\z/
    end

    def convert_S3_to_nginx_url(url)
      url = url.sub("https://leviathan-prod.s3-us-west-2.amazonaws.com/", "https://leviathan.moderation.twitch.tv/")
      url = url.sub("https://s3-us-west-2.amazonaws.com/leviathan-prod/", "https://leviathan.moderation.twitch.tv/")
    end

    def content_type_based_on_ext(ext)
      ext = ext.downcase
      if ext == "jpg" || ext == "jpeg"
        return "image/jpeg"
      elsif ext == "png"
        return "image/png"
      elsif ext == "gif"
        return "image/gif"
      end

      return "application/octet-stream"
    end
end

