class JanitorController < ApplicationController
  # Let only people with use_janitor permission do stuff
  before_action :require_use_janitor_permission
  # Let only people with view_janitor_audits view audits
  before_action :require_view_janitor_audits_permission, only: [:audits]


  def index
  end

  def clear_chat_rules
    if request.post?
      # Make sure all necessary parts are here
      unless params[:channel] && !params[:channel].blank?
        flash[:danger] = "No channel submitted"
        redirect_to(janitor_clear_chat_rules_path) and return
      end

      # Resolve channel => ID
      begin
          target_user = ::TwitchUsersService.get_user_by_username(params[:channel])

          if target_user.nil?
            flash[:danger] = "Username not found"
            redirect_to(janitor_clear_chat_rules_path) and return
          end
        rescue Exception => e
          flash[:danger] = "Error resolving the username"
          redirect_to(janitor_clear_chat_rules_path) and return
      end

      unless target_user && target_user["id"]
        flash[:danger] = "Username not found"
        redirect_to(janitor_clear_chat_rules_path) and return
      end

      # Do the request to tmi
      clear_chat_rules_from_site(target_user["id"])

      # Audit it
      audit = {}
      audit[:channel_id] = target_user["id"]
      audit[:channel_name] = target_user["username"]
      JanitorAudit.new(action: "clear_chat_rules", action_by: current_user.id, content: audit.to_json, remote_ip: request.remote_ip).save

      flash.now[:success] = "Chat rules successfully cleared on channel: \"#{target_user["username"]}\" (#{target_user["id"]})!"
    end
  end

  def clear_panels
    if request.post?
      # Make sure all necessary parts are here
      unless params[:channel] && !params[:channel].blank?
        flash[:danger] = "No channel submitted"
        redirect_to(janitor_clear_panels_path) and return
      end

      # Resolve channel => ID
      begin
          target_user = ::TwitchUsersService.get_user_by_username(params[:channel])

          if target_user.nil?
            flash[:danger] = "Username not found"
            redirect_to(janitor_clear_panels_path) and return
          end
        rescue Exception => e
          flash[:danger] = "Error resolving the username"
          redirect_to(janitor_clear_panels_path) and return
      end

      unless target_user && target_user["id"]
        flash[:danger] = "Username not found"
        redirect_to(janitor_clear_panels_path) and return
      end

      # Do the request to tmi
      @cleared_panel_count = 0
      clear_panels_from_site(target_user["id"])

      # Audit it
      audit = {}
      audit[:channel_id] = target_user["id"]
      audit[:channel_name] = target_user["username"]
      audit[:cleared_panel_count] = @cleared_panel_count
      JanitorAudit.new(action: "clear_panels", action_by: current_user.id, content: audit.to_json, remote_ip: request.remote_ip).save

      flash.now[:success] = "Successfully cleared #{@cleared_panel_count} panels on channel: \"#{target_user["username"]}\" (#{target_user["id"]})!"
    end
  end

  def vod_strike
    if params[:vod_url]
      # Make sure it's a valid VOD url
      if vod_url_parsed = params[:vod_url].match(/\Ahttps:\/\/www(\-origin)?\.twitch\.tv\/([a-z0-9_]+)\/v\/([0-9]+)/i)
        @strike_url = "#{Settings.twitch_base_url}#{vod_url_parsed[2].downcase}/report_form?tos_ban=true&content=vod&vod_id=#{vod_url_parsed[3]}"
        if params[:option] && params[:option] == "redirect"
          redirect_to(@strike_url) and return
        end
      else
        flash.now[:danger] = "Invalid VOD URL submitted"
      end
    end
  end

  def id_to_login
    if params[:user_id] && !params[:user_id].blank?
      # Check if it's a valid user_id (aka int-able)
      user_id = Integer(params[:user_id].strip) rescue false
      if user_id
        begin
          @user = ::TwitchUsersService.get_user_by_id(user_id)
          rescue Exception => e
            flash.now[:danger] = "Error while trying to lookup this User ID, maybe it doesn't exist?"
        end
      else
        flash.now[:danger] = "Invalid User ID submitted"
      end
    end
  end

  def audits
    @janitor_audits = JanitorAudit.order(id: :desc).paginate(page: params[:page])
  end

  private

    def clear_chat_rules_from_site(user_id)
      begin
          url = URI.parse("http://tmi-nginx-internal.production.us-west2.twitch.tv/rooms/#{user_id}")
          http = Net::HTTP.new(url.host, url.port)
          http.use_ssl = true if url.scheme == "https"
          http.open_timeout = 3
          http.read_timeout = 10
          req = Net::HTTP::Put.new(url.request_uri)
          req.add_field("Host", "tmi-clue.internal.twitch.tv")
          req.add_field("Content-Type", "application/json")
          req.body = '{"chat_rules":[]}'
          res = http.request(req)

          unless res.is_a?(Net::HTTPSuccess)
            flash[:danger] = "Error while clearing chat rules"
            redirect_to(janitor_clear_chat_rules_path) and return
          end
        rescue Exception => e
          flash[:danger] = "Error while clearing chat rules"
          redirect_to(janitor_clear_chat_rules_path) and return
      end
    end

    def clear_panels_from_site(user_id)
      # Get a list of all panels the user has
      begin
          url = URI.parse("http://panels.production.us-west2.twitch.tv/v1/users/#{user_id}/panels")
          http = Net::HTTP.new(url.host, url.port)
          http.use_ssl = true if url.scheme == "https"
          http.open_timeout = 3
          http.read_timeout = 10
          req = Net::HTTP::Get.new(url.request_uri)
          res = http.request(req)

          unless res.is_a?(Net::HTTPSuccess)
            flash[:danger] = "Error while getting panel overview"
            redirect_to(janitor_clear_panels_path) and return
          end
        rescue Exception => e
          flash[:danger] = "Error while getting panel overview"
          redirect_to(janitor_clear_panels_path) and return
      end

      # Parse result
      panel_list = JSON.parse(res.body)

      # Check if there are any panels
      if panel_list["panels"] && panel_list["panels"].any?
        # Delete each panel one by one
        panel_list["panels"].each do |panel|
          begin
              url = URI.parse("http://panels.production.us-west2.twitch.tv/v1/panels/#{panel["id"]}")
              http = Net::HTTP.new(url.host, url.port)
              http.use_ssl = true if url.scheme == "https"
              http.open_timeout = 3
              http.read_timeout = 10
              req = Net::HTTP::Delete.new(url.request_uri)
              res = http.request(req)

              unless res.is_a?(Net::HTTPSuccess)
                flash[:danger] = "Error while clearing panels"
                redirect_to(janitor_clear_panels_path) and return
              end
            rescue Exception => e
              flash[:danger] = "Error while clearing panels"
              redirect_to(janitor_clear_panels_path) and return
          end
          @cleared_panel_count = @cleared_panel_count + 1
        end
      end
    end

    def require_use_janitor_permission
      unless current_user && current_user.has_permission_to?(:use_janitor)
        render 'shared/_no_permission' and return
      end
    end

    def require_view_janitor_audits_permission
      unless current_user && current_user.has_permission_to?(:view_janitor_audits)
        render 'shared/_no_permission' and return
      end
    end
end
