class ReportHoldsController < ApplicationController
  # Let only people with view_reports permission do stuff
  before_action :require_view_reports_permission
  # Always return the current ReportVersion via the header
  after_action :return_report_version_via_header

  def show
    @report_hold = ReportHold.find_by(id: params[:id])
  end

  def create
    hold = ReportHold.new(hold_create_params)

    # Check if the report_id exists
    report_exists = Report.find_by(id: hold.report_id)
    unless report_exists
      render(json: { status: 404, message: "This report doesn't exist"}, :status => 200) and return
    end

    hold_exists = ReportHold.where(
                                   "hold_until > ? AND disabled_at IS NULL",
                                    Time.now
                                  ).where(
                                    report_id: hold.report_id
                                  )
    if hold_exists.count > 0
      render(json: { status: 422, message: "You are already holding this report"}, :status => 200) and return
    end


    if params[:duration] && params[:duration].to_i > 0 && params[:duration].to_i <= 3600
      hold.hold_until = Time.now().advance(seconds: params[:duration].to_i)
      hold.created_by = current_user.id

      if hold.save
        # All went well and hold is created
        # Audit it
        ReportAudit.new(
            report_id: hold.report_id,
            action: "hold",
            action_by: current_user.id,
            hold_id: hold.id,
        ).save

        # Output info
        render(json: { status: 200, message: "Hold successfully created!", holds: active}, :status => 200) and return
      else
        # Some error
        render(json: { status: 500, message: "You broke it"}, :status => 200) and return
      end
    else
      # No duration or not within limits
      render(json: { status: 400, message: "The submitted duration is not valid"}, :status => 200) and return
    end
  end

  def update
    hold = ReportHold.find_by(id: params[:id])
    if hold
      # Check action type
      if params[:do_action] && params[:do_action] == "disable"
        # Check if it's still active
        if hold.hold_until > Time.now && hold.disabled_at.nil?
          hold.disabled_at = Time.now
          if hold.save
            # Audit it
            ReportAudit.new(
              report_id: hold.report_id,
              action: "hold_disabled",
              action_by: current_user.id,
              hold_id: hold.id,
            ).save

            render(json: { status: 200, message: "Hold successfully released!", holds: active}, :status => 200) and return
          else
            render(json: { status: 500, message: "You broke it"}, :status => 200) and return
          end
        else
          render(json: { status: 404, message: "This hold isn't active anyway"}, :status => 200) and return
        end
      else
        render(json: { status: 422, message: "Unknown action"}, :status => 200) and return
      end
    else
      render(json: { status: 404, message: "This hold doesn't exist"}, :status => 200) and return
    end
  end


  private

    def hold_create_params
      params.require(:report_hold).permit(
        :report_id
        )
    end

    def active
      active_holds = []
      db_active_holds = ReportHold.where("hold_until > ? AND disabled_at IS NULL", Time.now)

      # Get all admin emails for the IDs
      if db_active_holds.length > 0
        user_ids = []
        db_active_holds.each do |h|
          user_ids << h.created_by
        end
        user_ids = user_ids.uniq

        admins = User.where(id: user_ids).pluck(:id, :email)
        admin_emails = {}
        admins.each do |admin|
          admin_emails[admin[0]] = admin[1]
        end
      end

      # Get report info for the hold reports
      if db_active_holds.length > 0
        report_ids = []
        db_active_holds.each do |h|
          report_ids << h.report_id
        end
        report_ids = report_ids.uniq

        reports = Report.where(id: report_ids).pluck(:id, :target_user_username, :content)
        reports_info = {}
        reports.each do |report|
          reports_info[report[0]] = { target_user_username: report[1], content: report[2] }
        end
      end

      db_active_holds.each do |h|
        active_holds << {
          id: h.id,
          report_id: h.report_id,
          target_user_username: reports_info[h.report_id][:target_user_username],
          hold_until: h.hold_until,
          content: reports_info[h.report_id][:content],
          created_by: h.created_by,
          created_by_username: admin_emails[h.created_by]
        }
        end

      active_holds
    end

    def require_view_reports_permission
      unless current_user && current_user.has_permission_to?(:view_reports)
        render 'shared/_no_permission' and return
      end
    end

    def return_report_version_via_header
      redis_report_version = $redis.get("reports:report_version")
      response.headers["X-Report-Version"] = redis_report_version if redis_report_version
      return true
    end
end
