class SuspensionGuideContentsController < ApplicationController
  # Let only people with manage_suspension_guide permission change stuff
  before_action :require_manage_suspension_guide, except: [:index, :show]

  def index
    @suspension_guide_contents = SuspensionGuideContent.order(content: :asc)
  end

  def show
    @suspension_guide_content = SuspensionGuideContent.find_by(id: params[:id])
  end

  def new
    @suspension_guide_content = SuspensionGuideContent.new
  end

  def edit
    @suspension_guide_content = SuspensionGuideContent.find_by(id: params[:id])
  end

  def create
    suspension_guide_content = SuspensionGuideContent.new(suspension_guide_content_params)

    if suspension_guide_content.save
      # All went well
      flash[:success] = "The suspension guide content was successfully added."

      # Audit it
      SuspensionGuideAudit.new(
        action: "new_content",
        action_by: current_user.id,
        diff: suspension_guide_content.attributes.except("created_at", "updated_at").to_json
      ).save

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error adding the suspension guide content: #{suspension_guide_content.errors.full_messages.to_sentence}"
    end

    redirect_to(suspension_guide_contents_path)
  end

  def update
    suspension_guide_content = SuspensionGuideContent.find_by(id: params[:id])
    suspension_guide_content_before_change = suspension_guide_content.dup

    if suspension_guide_content.update_attributes(suspension_guide_content_params)
      # All went well
      flash[:success] = "The suspension guide content was successfully updated."

      if suspension_guide_content_before_change.diff(suspension_guide_content).any?
        # Audit it
        SuspensionGuideAudit.new(
          action: "edit_content",
          action_by: current_user.id,
          diff: { id: suspension_guide_content.id }.merge(suspension_guide_content_before_change.diff(suspension_guide_content).to_hash).to_json
        ).save
      end

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error updating the suspension guide content: #{suspension_guide_content.errors.full_messages.to_sentence}"
    end

    redirect_to(edit_suspension_guide_content_path(suspension_guide_content))
  end

  def destroy
    suspension_guide_content = SuspensionGuideContent.find_by(id: params[:id])
    suspension_guide_content_before_change = suspension_guide_content.dup

    # Check if the content is used in the Suspension Guide
    suspension_guide_usage = SuspensionGuide.where(content_id: suspension_guide_content.id)
    if suspension_guide_usage.any?
      # You can't delete a content if the Suspension Guide uses it
      flash[:danger] = "You can't delete this content because it's currently used by the suspension guide."
    else
      if suspension_guide_content.destroy
        # All went well
        flash[:success] = "The suspension guide content was successfully removed."

        # Audit it
        SuspensionGuideAudit.new(
          action: "delete_content",
          action_by: current_user.id,
          diff: { id: params[:id].to_i }.reverse_merge(suspension_guide_content_before_change.attributes.except("created_at", "updated_at").to_hash).to_json
        ).save

        # Delete our cached version
        $redis.del("SuspensionGuide:addon_json")
        $redis.del("SuspensionGuide:addon_quick_selects_json")
      else
        # Some error
        flash[:danger] = "There was an error removing the suspension guide content: #{suspension_guide_content.errors.full_messages.to_sentence}"
      end
    end

    redirect_to(suspension_guide_contents_path)
  end


  private

    def suspension_guide_content_params
      params.require(:suspension_guide_content).permit(:content, :content_description)
    end

    def require_manage_suspension_guide
      unless current_user && current_user.has_permission_to?(:manage_suspension_guide)
        render 'shared/_no_permission' and return
      end
    end
end
