class SuspensionGuidesController < ApplicationController
  # Let only people with manage_suspension_guide permission change stuff
  before_action :require_manage_suspension_guide, except: [:index, :show, :addon, :audits]
  # Disable login check for js addon
  skip_before_action :require_login, only: [:addon]
  # Disable CSRF token check for js addon
  skip_before_action :verify_authenticity_token, only: [:addon]

  def index
    if params[:order_by] && params[:order_by] == "detailed_reason"
      @suspension_guides = SuspensionGuide.joins("JOIN (suspension_guide_contents, suspension_guide_detailed_reasons) ON (suspension_guide_contents.id = suspension_guides.content_id AND suspension_guide_detailed_reasons.id = suspension_guides.detailed_reason_id)").order("suspension_guide_detailed_reasons.detailed_reason asc", "suspension_guide_contents.content_description asc")
    else
      @suspension_guides = SuspensionGuide.joins("JOIN suspension_guide_contents ON suspension_guide_contents.id = suspension_guides.content_id").order("suspension_guide_contents.content_description asc", "suspension_guides.sorting asc")
    end

    @suspension_guide_contents = {}
    @suspension_guide_reasons = {}
    @suspension_guide_detailed_reasons = {}

    # Get the info we need to be able to display the guide
    if @suspension_guides.length > 0
      content_ids = []
      reason_ids = []
      detailed_reason_ids = []
      @suspension_guides.each do |suspension_guide|
        content_ids << suspension_guide.content_id
        reason_ids << suspension_guide.reason_id
        detailed_reason_ids << suspension_guide.detailed_reason_id
      end
      content_ids = content_ids.uniq
      reason_ids = reason_ids.uniq
      detailed_reason_ids = detailed_reason_ids.uniq

      if content_ids.size > 0
        contents = SuspensionGuideContent.where(id: content_ids).pluck(:id, :content_description)
        contents.each do |content|
          @suspension_guide_contents[content[0]] = content[1]
        end
      end

      if reason_ids.size > 0
        reasons = SuspensionGuideReason.where(id: reason_ids).pluck(:id, :reason_description)
        reasons.each do |reason|
          @suspension_guide_reasons[reason[0]] = reason[1]
        end
      end

      if detailed_reason_ids.size > 0
        detailed_reasons = SuspensionGuideDetailedReason.where(id: detailed_reason_ids).pluck(:id, :detailed_reason)
        detailed_reasons.each do |detailed_reason|
          @suspension_guide_detailed_reasons[detailed_reason[0]] = detailed_reason[1]
        end
      end
    end
  end

  def show
    @suspension_guide = SuspensionGuide.find_by(id: params[:id])
  end

  def new
    @suspension_guide = SuspensionGuide.new
  end

  def edit
    @suspension_guide = SuspensionGuide.find_by(id: params[:id])
  end

  def create
    @suspension_guide = SuspensionGuide.new(suspension_guide_params)

    # Check if the provided content_id exists
    unless @suspension_guide[:content_id] && SuspensionGuideContent.find_by(id: @suspension_guide[:content_id])
      flash.now[:danger] = "The selected content doesn't exist."
      render 'new' and return
    end

    # Check if the provided reason_id exists
    unless @suspension_guide[:reason_id] && SuspensionGuideReason.find_by(id: @suspension_guide[:reason_id])
      flash.now[:danger] = "The selected reason doesn't exist."
      render 'new' and return
    end

    # Check if the provided detailed_reason_id exists
    unless @suspension_guide[:detailed_reason_id] && SuspensionGuideDetailedReason.find_by(id: @suspension_guide[:detailed_reason_id])
      flash.now[:danger] = "The selected detailed reason doesn't exist."
      render 'new' and return
    end

    if @suspension_guide.save
      # Update the sorting based on the ID
      @suspension_guide.sorting = @suspension_guide.id * 10000
      @suspension_guide.save

      # All went well
      flash[:success] = "New suspension guide entry successfully created."

      # Audit it
      SuspensionGuideAudit.new(
        action: "new_suspension_guide",
        action_by: current_user.id,
        diff: @suspension_guide.attributes.except("created_at", "updated_at").to_json
      ).save

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      flash.now[:danger] = "#{@suspension_guide.errors.full_messages.to_sentence}"
      render 'new' and return
    end

    redirect_to(suspension_guides_path)
  end

  def update
    suspension_guide = SuspensionGuide.find_by(id: params[:id])
    suspension_guide_before_change = suspension_guide.dup

    # Check if the provided content_id exists
    unless suspension_guide[:content_id] && SuspensionGuideContent.find_by(id: suspension_guide[:content_id])
      flash[:danger] = "The selected content doesn't exist."
      redirect_to(edit_suspension_guide_path(suspension_guide)) and return
    end

    # Check if the provided reason_id exists
    unless suspension_guide[:reason_id] && SuspensionGuideReason.find_by(id: suspension_guide[:reason_id])
      flash[:danger] = "The selected reason doesn't exist."
      redirect_to(edit_suspension_guide_path(suspension_guide)) and return
    end

    # Check if the provided detailed_reason_id exists
    unless suspension_guide[:detailed_reason_id] && SuspensionGuideDetailedReason.find_by(id: suspension_guide[:detailed_reason_id])
      flash[:danger] = "The selected detailed reason doesn't exist."
      redirect_to(edit_suspension_guide_path(suspension_guide)) and return
    end

    if suspension_guide.update_attributes(suspension_guide_params)
      # All went well
      flash[:success] = "The suspension guide entry was successfully updated."

      if suspension_guide_before_change.diff(suspension_guide).any?
        # Audit it
        SuspensionGuideAudit.new(
          action: "edit_suspension_guide",
          action_by: current_user.id,
          diff: { id: suspension_guide.id }.merge(suspension_guide_before_change.diff(suspension_guide).to_hash).to_json
        ).save
      end

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error updating the suspension guide entry: #{suspension_guide.errors.full_messages.to_sentence}"
    end

    redirect_to(edit_suspension_guide_path(suspension_guide))
  end

  def destroy
    suspension_guide = SuspensionGuide.find_by(id: params[:id])
    suspension_guide_before_change = suspension_guide.dup

    if suspension_guide.destroy
      # All went well
      flash[:success] = "The suspension guide entry was successfully removed."

      # Audit it
      SuspensionGuideAudit.new(
        action: "delete_suspension_guide",
        action_by: current_user.id,
        diff: { id: params[:id].to_i }.reverse_merge(suspension_guide_before_change.attributes.except("created_at", "updated_at").to_hash).to_json
      ).save

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error removing the suspension guide entry reason: #{suspension_guide_detailed_reason.errors.full_messages.to_sentence}"
    end

    redirect_to(suspension_guides_path)
  end

  def addon
    # Check if we have a cached version redis
    @addon_guide_json = $redis.get("SuspensionGuide:addon_json")
    unless @addon_guide_json
      # Get the info from the DB
      suspension_guides = SuspensionGuide.joins("JOIN (suspension_guide_contents, suspension_guide_reasons) ON (suspension_guide_contents.id = suspension_guides.content_id AND suspension_guide_reasons.id = suspension_guides.reason_id)").where(ignore: false).order("suspension_guide_contents.content_description asc", "suspension_guide_reasons.reason_description asc", "suspension_guides.sorting asc")
      addon_guide = Hash.new{ |h,k| h[k] = Hash.new(&h.default_proc) }

      # Generate the format we need for the better reason script
      suspension_guides.each do |suspension_guide_entry|
        addon_guide[SuspensionGuideContent.find_by(id: suspension_guide_entry.content_id).content]["reasons"][SuspensionGuideReason.find_by(id: suspension_guide_entry.reason_id).reason]["detailed_reasons"][SuspensionGuideDetailedReason.find_by(id: suspension_guide_entry.detailed_reason_id).detailed_reason] = {'title' => suspension_guide_entry.detailed_reason_title,
          'report_details' => suspension_guide_entry.strike_details, 'ip_ban' => suspension_guide_entry.ip_ban, 'permanent' => suspension_guide_entry.permanent,
          'clear_images' => suspension_guide_entry.clear_images, 'strike_form_note' => suspension_guide_entry.strike_form_note, 'remove_offending_content' => suspension_guide_entry.remove_offending_content}

        # Base Content & Reason
        addon_guide[SuspensionGuideContent.find_by(id: suspension_guide_entry.content_id).content]['reasons'][SuspensionGuideReason.find_by(id: suspension_guide_entry.reason_id).reason]['description'] = SuspensionGuideReason.find_by(id: suspension_guide_entry.reason_id).reason_description
        addon_guide[SuspensionGuideContent.find_by(id: suspension_guide_entry.content_id).content]['description'] = SuspensionGuideContent.find_by(id: suspension_guide_entry.content_id).content_description
      end

      # Convert to json
      @addon_guide_json = addon_guide.to_json

      # Cache "json" in redis
      $redis.set("SuspensionGuide:addon_json", @addon_guide_json)
    end

    @addon_quick_selects_json = $redis.get("SuspensionGuide:addon_quick_selects_json")
    unless @addon_quick_selects_json
      quick_selects_array = []
      # Get the quick select reasons
      quick_selects = SuspensionGuide.joins("JOIN suspension_guide_contents ON suspension_guide_contents.id = suspension_guides.content_id").where(ignore: false, list_in_quick_select: true).order("suspension_guide_contents.content_description asc", "suspension_guides.sorting asc")
      quick_selects.each do |quick_select|
        quick_selects_array << {
          content: SuspensionGuideContent.find_by(id: quick_select.content_id).content,
          reason: SuspensionGuideReason.find_by(id: quick_select.reason_id).reason,
          detailed_reason: SuspensionGuideDetailedReason.find_by(id: quick_select.detailed_reason_id).detailed_reason,
          detailed_reason_title: quick_select.detailed_reason_title
        }
      end

      # Convert to json
      @addon_quick_selects_json = quick_selects_array.to_json

      # Cache "json" in redis
      $redis.set("SuspensionGuide:addon_quick_selects_json", @addon_quick_selects_json)
    end
  end

  def audits
    @suspension_guide_audits = SuspensionGuideAudit.order(id: :desc).paginate(page: params[:page])
  end


  private

    def suspension_guide_params
      params.require(:suspension_guide).permit(
        :ignore,
        :sorting,
        :content_id,
        :reason_id,
        :detailed_reason_id,
        :detailed_reason_title,
        :detailed_reason_description,
        :strike_details,
        :ip_ban,
        :permanent,
        :clear_images,
        :strike_form_note,
        :remove_offending_content,
        :escalate,
        :offense_send_to_user,
        :violated_tos_section,
        :list_in_quick_select
      )
    end

    def require_manage_suspension_guide
      unless current_user && current_user.has_permission_to?(:manage_suspension_guide)
        render 'shared/_no_permission' and return
      end
    end
end
