package resolver

import (
	"fmt"
	"testing"

	"github.com/graph-gophers/graphql-go"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func newResolverTest() *resolverTest {
	comments := new(mockCommentsAPI)
	posts := new(mockPostsAPI)
	sponsoredPosts := new(mockSponsoredPostsAPI)
	timelineEntries := new(mockTimelineEntriesAPI)
	users := new(mockUsersAPI)
	return &resolverTest{
		Resolver: &Resolver{
			comments:        comments,
			posts:           posts,
			sponsoredPosts:  sponsoredPosts,
			timelineEntries: timelineEntries,
			users:           users,
		},
		comments:        comments,
		posts:           posts,
		sponsoredPosts:  sponsoredPosts,
		timelineEntries: timelineEntries,
		users:           users,
	}
}

type resolverTest struct {
	Resolver        *Resolver
	comments        *mockCommentsAPI
	posts           *mockPostsAPI
	sponsoredPosts  *mockSponsoredPostsAPI
	timelineEntries *mockTimelineEntriesAPI
	users           *mockUsersAPI
}

func (rt *resolverTest) Teardown(t *testing.T) {
	rt.comments.AssertExpectations(t)
	rt.posts.AssertExpectations(t)
	rt.sponsoredPosts.AssertExpectations(t)
	rt.timelineEntries.AssertExpectations(t)
	rt.users.AssertExpectations(t)
}

func TestResolver(t *testing.T) {
	const first = 23
	const after = "after"

	gqlIDPointer := func(in string) *graphql.ID {
		g := graphql.ID(in)
		return &g
	}

	stringPointer := func(in string) *string {
		return &in
	}

	t.Run("Posts", func(t *testing.T) {
		rt := newResolverTest()
		defer rt.Teardown(t)

		posts := make(posts, 0)
		rt.posts.On("After", gqlIDPointer(after)).Return(posts)

		assert.Equal(t, []*post(posts), rt.Resolver.Posts(struct {
			First int32
			After *graphql.ID
		}{First: first, After: gqlIDPointer(after)}))
	})

	t.Run("SponsoredPosts", func(t *testing.T) {
		rt := newResolverTest()
		defer rt.Teardown(t)

		sponsoredPosts := make(sponsoredPosts, 0)
		rt.sponsoredPosts.On("After", gqlIDPointer(after)).Return(sponsoredPosts)

		assert.Equal(t, []*sponsoredPost(sponsoredPosts), rt.Resolver.SponsoredPosts(struct {
			First int32
			After *graphql.ID
		}{First: first, After: gqlIDPointer(after)}))
	})

	t.Run("Timeline", func(t *testing.T) {
		rt := newResolverTest()
		defer rt.Teardown(t)

		timelineEntries := make(timelineEntries, 0)
		rt.timelineEntries.On("After", gqlIDPointer(after)).Return(timelineEntries)

		assert.Equal(t, []*timelineEntry(timelineEntries), rt.Resolver.Timeline(struct {
			First int32
			After *graphql.ID
		}{First: first, After: gqlIDPointer(after)}))
	})

	t.Run("UpdateUser", func(t *testing.T) {
		const userID = "userid"
		const userName = "username"

		t.Run("does not exist", func(t *testing.T) {
			rt := newResolverTest()
			defer rt.Teardown(t)

			rt.users.On("Get", graphql.ID(userID)).Return(nil, false)

			_, err := rt.Resolver.UpdateUser(struct {
				ID   graphql.ID
				Name *string
			}{
				ID: graphql.ID(userID),
			})
			assert.Equal(t, fmt.Errorf("user<%s> does not exist", userID), err)
		})

		t.Run("exists with username update", func(t *testing.T) {
			rt := newResolverTest()
			defer rt.Teardown(t)

			u := &user{}

			rt.users.On("Get", graphql.ID(userID)).Return(u, true)

			res, err := rt.Resolver.UpdateUser(struct {
				ID   graphql.ID
				Name *string
			}{
				ID:   graphql.ID(userID),
				Name: stringPointer(userName),
			})
			require.NoError(t, err)
			assert.Equal(t, u, res)
			assert.Equal(t, userName, u.Name())
		})

		t.Run("exists with no username update", func(t *testing.T) {
			rt := newResolverTest()
			defer rt.Teardown(t)

			u := &user{}

			rt.users.On("Get", graphql.ID(userID)).Return(u, true)

			res, err := rt.Resolver.UpdateUser(struct {
				ID   graphql.ID
				Name *string
			}{
				ID: graphql.ID(userID),
			})
			require.NoError(t, err)
			assert.Equal(t, u, res)
			assert.Equal(t, "", u.Name())
		})
	})
}
