package resolver

import (
	"fmt"
	"testing"

	"github.com/graph-gophers/graphql-go"
	"github.com/stretchr/testify/assert"
)

func TestTimelineEntry(t *testing.T) {
	const id = "id"

	posts := new(mockPostsAPI)
	sponsoredPosts := new(mockSponsoredPostsAPI)
	te := &timelineEntry{
		fields: TimelineEntryFields{
			ID: graphql.ID(id),
		},
		posts:          posts,
		sponsoredPosts: sponsoredPosts,
	}

	assert.Equal(t, graphql.ID(id), te.ID())

	tcs := []struct {
		TimelineEntry   *timelineEntry
		Post            *post
		IsPost          bool
		SponsoredPost   *sponsoredPost
		IsSponsoredPost bool
	}{
		{
			TimelineEntry: &timelineEntry{
				fields: TimelineEntryFields{
					ID:        graphql.ID(id),
					EntryType: PostTimelineEntryType,
				},
			},
			Post: &post{
				fields: PostFields{
					ID: graphql.ID(id),
				},
			},
			IsPost: true,
		},
		{
			TimelineEntry: &timelineEntry{
				fields: TimelineEntryFields{
					ID:        graphql.ID(id),
					EntryType: SponsoredPostTimelineEntryType,
				},
			},
			SponsoredPost: &sponsoredPost{
				fields: SponsoredPostFields{
					ID: graphql.ID(id),
				},
			},
			IsSponsoredPost: true,
		},
	}

	for nTc, tc := range tcs {
		t.Run(fmt.Sprintf("%d", nTc), func(t *testing.T) {
			posts := new(mockPostsAPI)
			defer posts.AssertExpectations(t)

			sponsoredPosts := new(mockSponsoredPostsAPI)
			defer sponsoredPosts.AssertExpectations(t)

			tc.TimelineEntry.posts = posts
			tc.TimelineEntry.sponsoredPosts = sponsoredPosts

			if tc.IsPost {
				posts.On("Get", graphql.ID(id)).Return(tc.Post, true)
			}
			rp, rpOK := tc.TimelineEntry.ToPost()
			assert.Equal(t, tc.IsPost, rpOK)
			assert.Equal(t, tc.Post, rp)

			if tc.IsSponsoredPost {
				sponsoredPosts.On("Get", graphql.ID(id)).Return(tc.SponsoredPost, true)
			}
			rsp, rspOK := tc.TimelineEntry.ToSponsoredPost()
			assert.Equal(t, tc.IsSponsoredPost, rspOK)
			assert.Equal(t, tc.SponsoredPost, rsp)
		})
	}
}
